/*
 * Decompiled with CFR 0.152.
 */
package org.popcraft.bolt.data;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.LogManager;
import org.popcraft.bolt.access.AccessList;
import org.popcraft.bolt.data.Store;
import org.popcraft.bolt.data.sql.Statements;
import org.popcraft.bolt.protection.BlockProtection;
import org.popcraft.bolt.protection.EntityProtection;
import org.popcraft.bolt.util.BlockLocation;
import org.popcraft.bolt.util.Group;
import org.popcraft.bolt.util.Metrics;

public class SQLStore
implements Store {
    private static final Gson GSON = new Gson();
    private static final TypeToken<HashMap<String, String>> ACCESS_LIST_TYPE_TOKEN = new TypeToken<HashMap<String, String>>(){};
    private static final TypeToken<List<String>> PLAYER_LIST_TYPE_TOKEN = new TypeToken<List<String>>(){};
    private static final Type ACCESS_LIST_TYPE = ACCESS_LIST_TYPE_TOKEN.getType();
    private static final Type PLAYER_LIST_TYPE = PLAYER_LIST_TYPE_TOKEN.getType();
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final Map<UUID, BlockProtection> saveBlocks = new HashMap<UUID, BlockProtection>();
    private final Map<UUID, BlockProtection> removeBlocks = new HashMap<UUID, BlockProtection>();
    private final Map<UUID, EntityProtection> saveEntities = new HashMap<UUID, EntityProtection>();
    private final Map<UUID, EntityProtection> removeEntities = new HashMap<UUID, EntityProtection>();
    private final Map<String, Group> saveGroups = new HashMap<String, Group>();
    private final Map<String, Group> removeGroups = new HashMap<String, Group>();
    private final Map<UUID, AccessList> saveAccessLists = new HashMap<UUID, AccessList>();
    private final Map<UUID, AccessList> removeAccessLists = new HashMap<UUID, AccessList>();
    private final Configuration configuration;
    private final String connectionUrl;
    private Connection connection;

    public SQLStore(Configuration configuration) {
        boolean usingMySQL;
        this.configuration = configuration;
        if ("sqlite".equals(configuration.type())) {
            try {
                Files.createDirectories(Path.of(".", new String[0]).resolve(configuration.path()).getParent(), new FileAttribute[0]);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }
        this.connectionUrl = (usingMySQL = "mysql".equals(configuration.type())) ? "jdbc:mysql://%s/%s".formatted(configuration.hostname(), configuration.database()) : "jdbc:sqlite:%s".formatted(configuration.path());
        this.reconnect();
        try (PreparedStatement createBlocksTable = this.connection.prepareStatement(Statements.CREATE_TABLE_BLOCKS.get(configuration.type()).formatted(configuration.prefix()));
             PreparedStatement createEntitiesTable = this.connection.prepareStatement(Statements.CREATE_TABLE_ENTITIES.get(configuration.type()).formatted(configuration.prefix()));
             PreparedStatement createGroupsTable = this.connection.prepareStatement(Statements.CREATE_TABLE_GROUPS.get(configuration.type()).formatted(configuration.prefix()));
             PreparedStatement createAccessTable = this.connection.prepareStatement(Statements.CREATE_TABLE_ACCESS.get(configuration.type()).formatted(configuration.prefix()));){
            createBlocksTable.execute();
            createEntitiesTable.execute();
            createGroupsTable.execute();
            createAccessTable.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
        if (!usingMySQL) {
            try (PreparedStatement createBlocksOwnerIndex = this.connection.prepareStatement(Statements.CREATE_INDEX_BLOCK_OWNER.get(configuration.type()).formatted(configuration.prefix()));
                 PreparedStatement createBlocksLocationIndex = this.connection.prepareStatement(Statements.CREATE_INDEX_BLOCK_LOCATION.get(configuration.type()).formatted(configuration.prefix()));
                 PreparedStatement createEntitiesOwnerIndex = this.connection.prepareStatement(Statements.CREATE_INDEX_ENTITY_OWNER.get(configuration.type()).formatted(configuration.prefix()));
                 PreparedStatement createGroupsOwnerIndex = this.connection.prepareStatement(Statements.CREATE_INDEX_GROUP_OWNER.get(configuration.type()).formatted(configuration.prefix()));){
                createBlocksOwnerIndex.execute();
                createBlocksLocationIndex.execute();
                createEntitiesOwnerIndex.execute();
                createGroupsOwnerIndex.execute();
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
        }
        this.executor.scheduleWithFixedDelay(this::flush, 30L, 30L, TimeUnit.SECONDS);
        if (usingMySQL) {
            this.executor.scheduleWithFixedDelay(this::reconnect, 30L, 30L, TimeUnit.MINUTES);
        }
    }

    private void reconnect() {
        try {
            if (this.connection != null) {
                this.connection.close();
            }
            this.connection = DriverManager.getConnection(this.connectionUrl, this.configuration.username(), this.configuration.password());
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void close() {
        this.executor.close();
        try {
            this.connection.close();
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public CompletableFuture<BlockProtection> loadBlockProtection(BlockLocation location) {
        CompletableFuture<BlockProtection> future = new CompletableFuture<BlockProtection>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectBlock = this.connection.prepareStatement(Statements.SELECT_BLOCK_BY_LOCATION.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                selectBlock.setString(1, location.world());
                selectBlock.setInt(2, location.x());
                selectBlock.setInt(3, location.y());
                selectBlock.setInt(4, location.z());
                ResultSet blockResultSet = selectBlock.executeQuery();
                if (blockResultSet.next()) {
                    future.complete(this.blockProtectionFromResultSet(blockResultSet));
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            Metrics.recordProtectionAccess(false);
            future.complete(null);
        }, this.executor);
        return future;
    }

    @Override
    public CompletableFuture<Collection<BlockProtection>> loadBlockProtections() {
        CompletableFuture<Collection<BlockProtection>> future = new CompletableFuture<Collection<BlockProtection>>();
        CompletableFuture.runAsync(() -> {
            long startTimeNanos = System.nanoTime();
            long[] count = new long[1];
            try (PreparedStatement selectBlocks = this.connection.prepareStatement(Statements.SELECT_ALL_BLOCKS.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                ResultSet blocksResultSet = selectBlocks.executeQuery();
                ArrayList<BlockProtection> protections = new ArrayList<BlockProtection>();
                while (blocksResultSet.next()) {
                    protections.add(this.blockProtectionFromResultSet(blocksResultSet));
                    count[0] = count[0] + 1L;
                }
                long timeNanos = System.nanoTime() - startTimeNanos;
                double timeMillis = (double)timeNanos / 1000000.0;
                LogManager.getLogManager().getLogger("").info(() -> "Loaded %d block protections in %.3f ms".formatted(count[0], timeMillis));
                future.complete(protections);
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(Collections.emptyList());
        }, this.executor);
        return future;
    }

    private BlockProtection blockProtectionFromResultSet(ResultSet resultSet) throws SQLException {
        String id = resultSet.getString(1);
        String owner = resultSet.getString(2);
        String type = resultSet.getString(3);
        long created = resultSet.getLong(4);
        long accessed = resultSet.getLong(5);
        String accessText = resultSet.getString(6);
        Map access = Objects.requireNonNullElse((HashMap)GSON.fromJson(accessText, ACCESS_LIST_TYPE_TOKEN), new HashMap());
        String world = resultSet.getString(7);
        int x = resultSet.getInt(8);
        int y = resultSet.getInt(9);
        int z = resultSet.getInt(10);
        String block = resultSet.getString(11);
        return new BlockProtection(UUID.fromString(id), UUID.fromString(owner), type, created, accessed, access, world, x, y, z, block);
    }

    @Override
    public void saveBlockProtection(BlockProtection protection) {
        CompletableFuture.runAsync(() -> this.saveBlocks.put(protection.getId(), protection), this.executor);
    }

    private void saveBlockProtectionNow(BlockProtection protection) {
        try (PreparedStatement replaceBlock = this.connection.prepareStatement(Statements.REPLACE_BLOCK.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            replaceBlock.setString(1, protection.getId().toString());
            replaceBlock.setString(2, protection.getOwner().toString());
            replaceBlock.setString(3, protection.getType());
            replaceBlock.setLong(4, protection.getCreated());
            replaceBlock.setLong(5, protection.getAccessed());
            replaceBlock.setString(6, GSON.toJson(protection.getAccess(), ACCESS_LIST_TYPE));
            replaceBlock.setString(7, protection.getWorld());
            replaceBlock.setInt(8, protection.getX());
            replaceBlock.setInt(9, protection.getY());
            replaceBlock.setInt(10, protection.getZ());
            replaceBlock.setString(11, protection.getBlock());
            replaceBlock.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void removeBlockProtection(BlockProtection protection) {
        CompletableFuture.runAsync(() -> {
            UUID id = protection.getId();
            this.saveBlocks.remove(id);
            this.removeBlocks.put(id, protection);
        }, this.executor);
    }

    private void removeBlockProtectionNow(BlockProtection protection) {
        try (PreparedStatement deleteBlock = this.connection.prepareStatement(Statements.DELETE_BLOCK.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            deleteBlock.setString(1, protection.getId().toString());
            deleteBlock.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public CompletableFuture<EntityProtection> loadEntityProtection(UUID id) {
        CompletableFuture<EntityProtection> future = new CompletableFuture<EntityProtection>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectEntity = this.connection.prepareStatement(Statements.SELECT_ENTITY_BY_UUID.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                selectEntity.setString(1, id.toString());
                ResultSet entityResultSet = selectEntity.executeQuery();
                if (entityResultSet.next()) {
                    future.complete(this.entityProtectionFromResultSet(entityResultSet));
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            Metrics.recordProtectionAccess(false);
            future.complete(null);
        }, this.executor);
        return future;
    }

    @Override
    public CompletableFuture<Collection<EntityProtection>> loadEntityProtections() {
        CompletableFuture<Collection<EntityProtection>> future = new CompletableFuture<Collection<EntityProtection>>();
        CompletableFuture.runAsync(() -> {
            long startTimeNanos = System.nanoTime();
            long[] count = new long[1];
            try (PreparedStatement selectEntities = this.connection.prepareStatement(Statements.SELECT_ALL_ENTITIES.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                ResultSet entitiesResultSet = selectEntities.executeQuery();
                ArrayList<EntityProtection> protections = new ArrayList<EntityProtection>();
                while (entitiesResultSet.next()) {
                    protections.add(this.entityProtectionFromResultSet(entitiesResultSet));
                    count[0] = count[0] + 1L;
                }
                long timeNanos = System.nanoTime() - startTimeNanos;
                double timeMillis = (double)timeNanos / 1000000.0;
                LogManager.getLogManager().getLogger("").info(() -> "Loaded %d entity protections in %.3f ms".formatted(count[0], timeMillis));
                future.complete(protections);
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(Collections.emptyList());
        }, this.executor);
        return future;
    }

    private EntityProtection entityProtectionFromResultSet(ResultSet resultSet) throws SQLException {
        String id = resultSet.getString(1);
        String owner = resultSet.getString(2);
        String type = resultSet.getString(3);
        long created = resultSet.getLong(4);
        long accessed = resultSet.getLong(5);
        String accessText = resultSet.getString(6);
        Map access = Objects.requireNonNullElse((HashMap)GSON.fromJson(accessText, ACCESS_LIST_TYPE_TOKEN), new HashMap());
        String entity = resultSet.getString(7);
        return new EntityProtection(UUID.fromString(id), UUID.fromString(owner), type, created, accessed, access, entity);
    }

    @Override
    public void saveEntityProtection(EntityProtection protection) {
        CompletableFuture.runAsync(() -> this.saveEntities.put(protection.getId(), protection), this.executor);
    }

    private void saveEntityProtectionNow(EntityProtection protection) {
        try (PreparedStatement replaceEntity = this.connection.prepareStatement(Statements.REPLACE_ENTITY.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            replaceEntity.setString(1, protection.getId().toString());
            replaceEntity.setString(2, protection.getOwner().toString());
            replaceEntity.setString(3, protection.getType());
            replaceEntity.setLong(4, protection.getCreated());
            replaceEntity.setLong(5, protection.getAccessed());
            replaceEntity.setString(6, GSON.toJson(protection.getAccess(), ACCESS_LIST_TYPE));
            replaceEntity.setString(7, protection.getEntity());
            replaceEntity.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void removeEntityProtection(EntityProtection protection) {
        CompletableFuture.runAsync(() -> {
            this.saveEntities.remove(protection.getId());
            this.removeEntities.put(protection.getId(), protection);
        }, this.executor);
    }

    private void removeEntityProtectionNow(EntityProtection protection) {
        try (PreparedStatement deleteEntity = this.connection.prepareStatement(Statements.DELETE_ENTITY.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            deleteEntity.setString(1, protection.getId().toString());
            deleteEntity.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public CompletableFuture<Group> loadGroup(String group) {
        CompletableFuture<Group> future = new CompletableFuture<Group>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectGroup = this.connection.prepareStatement(Statements.SELECT_GROUP_BY_NAME.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                selectGroup.setString(1, group);
                ResultSet groupResultSet = selectGroup.executeQuery();
                if (groupResultSet.next()) {
                    future.complete(this.groupFromResultSet(groupResultSet));
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(null);
        }, this.executor);
        return future;
    }

    @Override
    public CompletableFuture<Collection<Group>> loadGroups() {
        CompletableFuture<Collection<Group>> future = new CompletableFuture<Collection<Group>>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectGroups = this.connection.prepareStatement(Statements.SELECT_ALL_GROUPS.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                ResultSet groupResultSet = selectGroups.executeQuery();
                ArrayList<Group> groups = new ArrayList<Group>();
                while (groupResultSet.next()) {
                    groups.add(this.groupFromResultSet(groupResultSet));
                }
                future.complete(groups);
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(Collections.emptyList());
        }, this.executor);
        return future;
    }

    private Group groupFromResultSet(ResultSet resultSet) throws SQLException {
        String name = resultSet.getString(1);
        String owner = resultSet.getString(2);
        String membersText = resultSet.getString(3);
        List membersRaw = Objects.requireNonNullElse((List)GSON.fromJson(membersText, PLAYER_LIST_TYPE_TOKEN), new ArrayList());
        ArrayList<UUID> members = new ArrayList<UUID>();
        membersRaw.forEach(memberRaw -> members.add(UUID.fromString(memberRaw)));
        return new Group(name, UUID.fromString(owner), members);
    }

    @Override
    public void saveGroup(Group group) {
        CompletableFuture.runAsync(() -> this.saveGroups.put(group.getName(), group), this.executor);
    }

    private void saveGroupNow(Group group) {
        try (PreparedStatement replaceGroup = this.connection.prepareStatement(Statements.REPLACE_GROUP.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            replaceGroup.setString(1, group.getName());
            replaceGroup.setString(2, group.getOwner().toString());
            replaceGroup.setString(3, GSON.toJson(group.getMembers(), PLAYER_LIST_TYPE));
            replaceGroup.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void removeGroup(Group group) {
        CompletableFuture.runAsync(() -> this.removeGroups.put(group.getName(), group), this.executor);
    }

    private void removeGroupNow(Group group) {
        try (PreparedStatement deleteGroup = this.connection.prepareStatement(Statements.DELETE_GROUP.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            deleteGroup.setString(1, group.getName());
            deleteGroup.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public CompletableFuture<AccessList> loadAccessList(UUID owner) {
        CompletableFuture<AccessList> future = new CompletableFuture<AccessList>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectAccessList = this.connection.prepareStatement(Statements.SELECT_ACCESS_LIST_BY_UUID.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                selectAccessList.setString(1, owner.toString());
                ResultSet accessListResultSet = selectAccessList.executeQuery();
                if (accessListResultSet.next()) {
                    future.complete(this.accessListFromResultSet(accessListResultSet));
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(null);
        }, this.executor);
        return future;
    }

    @Override
    public CompletableFuture<Collection<AccessList>> loadAccessLists() {
        CompletableFuture<Collection<AccessList>> future = new CompletableFuture<Collection<AccessList>>();
        CompletableFuture.runAsync(() -> {
            try (PreparedStatement selectAccessLists = this.connection.prepareStatement(Statements.SELECT_ALL_ACCESS_LISTS.get(this.configuration.type()).formatted(this.configuration.prefix()));){
                ResultSet accessListsResultSet = selectAccessLists.executeQuery();
                ArrayList<AccessList> accessLists = new ArrayList<AccessList>();
                while (accessListsResultSet.next()) {
                    accessLists.add(this.accessListFromResultSet(accessListsResultSet));
                }
                future.complete(accessLists);
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            future.complete(Collections.emptyList());
        }, this.executor);
        return future;
    }

    private AccessList accessListFromResultSet(ResultSet resultSet) throws SQLException {
        String owner = resultSet.getString(1);
        String accessListText = resultSet.getString(2);
        Map access = Objects.requireNonNullElse((HashMap)GSON.fromJson(accessListText, ACCESS_LIST_TYPE_TOKEN), new HashMap());
        return new AccessList(UUID.fromString(owner), access);
    }

    @Override
    public void saveAccessList(AccessList accessList) {
        CompletableFuture.runAsync(() -> this.saveAccessLists.put(accessList.getOwner(), accessList), this.executor);
    }

    private void saveAccessListNow(AccessList accessList) {
        try (PreparedStatement replaceAccessList = this.connection.prepareStatement(Statements.REPLACE_ACCESS_LIST.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            replaceAccessList.setString(1, accessList.getOwner().toString());
            replaceAccessList.setString(2, GSON.toJson(accessList.getAccess(), ACCESS_LIST_TYPE));
            replaceAccessList.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void removeAccessList(AccessList accessList) {
        CompletableFuture.runAsync(() -> this.removeAccessLists.put(accessList.getOwner(), accessList), this.executor);
    }

    private void removeAccessListNow(AccessList accessList) {
        try (PreparedStatement deleteAccessList = this.connection.prepareStatement(Statements.DELETE_ACCESS_LIST.get(this.configuration.type()).formatted(this.configuration.prefix()));){
            deleteAccessList.setString(1, accessList.getOwner().toString());
            deleteAccessList.execute();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    @Override
    public long pendingSave() {
        return CompletableFuture.supplyAsync(() -> this.saveBlocks.size() + this.removeBlocks.size() + this.saveEntities.size() + this.removeEntities.size(), this.executor).join().intValue();
    }

    @Override
    public CompletableFuture<Void> flush() {
        CompletableFuture<Void> completionFuture = new CompletableFuture<Void>();
        CompletableFuture.runAsync(() -> {
            try {
                if (!this.saveBlocks.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<BlockProtection> saveBlocksIterator = this.saveBlocks.values().iterator();
                    while (saveBlocksIterator.hasNext()) {
                        this.saveBlockProtectionNow(saveBlocksIterator.next());
                        saveBlocksIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.removeBlocks.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<BlockProtection> removeBlocksIterator = this.removeBlocks.values().iterator();
                    while (removeBlocksIterator.hasNext()) {
                        this.removeBlockProtectionNow(removeBlocksIterator.next());
                        removeBlocksIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.saveEntities.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<EntityProtection> saveEntitiesIterator = this.saveEntities.values().iterator();
                    while (saveEntitiesIterator.hasNext()) {
                        this.saveEntityProtectionNow(saveEntitiesIterator.next());
                        saveEntitiesIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.removeEntities.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<EntityProtection> removeEntitiesIterator = this.removeEntities.values().iterator();
                    while (removeEntitiesIterator.hasNext()) {
                        this.removeEntityProtectionNow(removeEntitiesIterator.next());
                        removeEntitiesIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.saveGroups.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<Group> saveGroupsIterator = this.saveGroups.values().iterator();
                    while (saveGroupsIterator.hasNext()) {
                        this.saveGroupNow(saveGroupsIterator.next());
                        saveGroupsIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.removeGroups.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<Group> removeGroupsIterator = this.removeGroups.values().iterator();
                    while (removeGroupsIterator.hasNext()) {
                        this.removeGroupNow(removeGroupsIterator.next());
                        removeGroupsIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.saveAccessLists.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<AccessList> saveAccessListsIterator = this.saveAccessLists.values().iterator();
                    while (saveAccessListsIterator.hasNext()) {
                        this.saveAccessListNow(saveAccessListsIterator.next());
                        saveAccessListsIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
                if (!this.removeAccessLists.isEmpty()) {
                    this.connection.setAutoCommit(false);
                    Iterator<AccessList> removeAccessListsIterator = this.removeAccessLists.values().iterator();
                    while (removeAccessListsIterator.hasNext()) {
                        this.removeAccessListNow(removeAccessListsIterator.next());
                        removeAccessListsIterator.remove();
                    }
                    this.connection.setAutoCommit(true);
                }
            }
            catch (SQLException e) {
                e.printStackTrace();
            }
            finally {
                completionFuture.complete(null);
            }
        }, this.executor);
        return completionFuture;
    }

    public record Configuration(String type, String path, String hostname, String database, String username, String password, String prefix, Map<String, String> properties) {
    }
}

