/*
 * Decompiled with CFR 0.152.
 */
package net.nando256.twbridge.http;

import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpServer;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.Base64;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.nando256.twbridge.TwBridgePlugin;

public final class TwHttpServer {
    private final TwBridgePlugin plugin;
    private final String address;
    private final int port;
    private final String path;
    private final List<String> corsAllowOrigins;
    private final int cacheSeconds;
    private final String wsDefault;
    private HttpServer server;
    private String jsTemplate;
    private final ConcurrentHashMap<String, JsVariant> variantCache = new ConcurrentHashMap();
    private static final Pattern WS_DEFAULT_PATTERN = Pattern.compile("const WS_DEFAULT = \"[^\"]+\";");
    private static final Pattern LANG_CONST_PATTERN = Pattern.compile("const TWB_DEFAULT_LANG = \"[^\"]*\";");
    private static final String BLOCK_LIST_PLACEHOLDER = "__TWB_BLOCK_CHOICES__";
    private static final Pattern LANG_SANITIZE_PATTERN = Pattern.compile("^[a-z0-9]{2,8}(?:-[a-z0-9]{1,8})*$");

    public TwHttpServer(TwBridgePlugin plugin, String address, int port, String path, List<String> corsAllowOrigins, int cacheSeconds, String wsDefault) {
        this.plugin = plugin;
        this.address = address;
        this.port = port;
        this.path = path == null || path.isBlank() ? "/tw/twbridge.js" : path;
        this.corsAllowOrigins = corsAllowOrigins;
        this.cacheSeconds = Math.max(0, cacheSeconds);
        this.wsDefault = wsDefault;
    }

    public void start() throws IOException {
        String blockListJson = this.buildBlockListJson();
        try (InputStream is = this.plugin.getResource("turbowarp/twbridge.js");){
            if (is == null) {
                throw new IOException("resource turbowarp/twbridge.js not found");
            }
            this.jsTemplate = this.loadTemplate(is, blockListJson);
        }
        this.variantCache.clear();
        this.server = HttpServer.create(new InetSocketAddress(this.address, this.port), 0);
        this.server.createContext("/", this::root);
        this.server.createContext(this.path, this::serveJs);
        this.server.setExecutor(Executors.newCachedThreadPool());
        this.server.start();
    }

    public void stop() {
        if (this.server != null) {
            this.server.stop(0);
            this.server = null;
        }
    }

    private void root(HttpExchange x) throws IOException {
        byte[] body = ("twbridge HTTP up.\nGET " + this.path + "\n").getBytes(StandardCharsets.UTF_8);
        this.setCommon(x.getResponseHeaders());
        x.sendResponseHeaders(200, body.length);
        try (OutputStream os = x.getResponseBody();){
            os.write(body);
        }
    }

    private void serveJs(HttpExchange x) throws IOException {
        this.setCommon(x.getResponseHeaders());
        x.getResponseHeaders().add("Content-Type", "text/javascript; charset=utf-8");
        x.getResponseHeaders().add("Cache-Control", "public, max-age=" + this.cacheSeconds);
        String langParam = this.extractLang(x.getRequestURI());
        JsVariant variant = this.variantCache.computeIfAbsent(langParam, this::buildVariant);
        x.getResponseHeaders().add("ETag", variant.etag());
        if (Objects.equals(x.getRequestMethod(), "OPTIONS")) {
            x.getResponseHeaders().add("Access-Control-Allow-Methods", "GET, HEAD, OPTIONS");
            x.getResponseHeaders().add("Access-Control-Allow-Headers", "Content-Type");
            x.sendResponseHeaders(204, -1L);
            x.close();
            return;
        }
        String inm = x.getRequestHeaders().getFirst("If-None-Match");
        if (inm != null && inm.equals(variant.etag()) && Objects.equals(x.getRequestMethod(), "GET")) {
            x.sendResponseHeaders(304, -1L);
            x.close();
            return;
        }
        boolean head = Objects.equals(x.getRequestMethod(), "HEAD");
        long len = variant.bytes().length;
        x.sendResponseHeaders(200, head ? -1L : len);
        if (!head) {
            try (OutputStream os = x.getResponseBody();){
                os.write(variant.bytes());
            }
        } else {
            x.close();
        }
    }

    private void setCommon(Headers h) {
        if (this.corsAllowOrigins != null && !this.corsAllowOrigins.isEmpty()) {
            h.add("Access-Control-Allow-Origin", this.corsAllowOrigins.get(0));
        }
        h.add("X-Content-Type-Options", "nosniff");
    }

    private static String calcEtag(byte[] d) {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            return "\"sha256-" + Base64.getUrlEncoder().withoutPadding().encodeToString(md.digest(d)) + "\"";
        }
        catch (Exception e) {
            return "\"" + d.length + "\"";
        }
    }

    private String loadTemplate(InputStream is, String blockListJson) throws IOException {
        Matcher matcher;
        String raw = new String(is.readAllBytes(), StandardCharsets.UTF_8);
        if (this.wsDefault != null && !this.wsDefault.isBlank() && (matcher = WS_DEFAULT_PATTERN.matcher(raw)).find()) {
            String safe = TwHttpServer.escapeForJs(this.wsDefault);
            raw = matcher.replaceFirst("const WS_DEFAULT = \"" + safe + "\";");
        }
        if (blockListJson != null) {
            raw = raw.replace(BLOCK_LIST_PLACEHOLDER, blockListJson);
        }
        return raw;
    }

    private JsVariant buildVariant(String requestedLang) {
        String lang = this.sanitizeLang(requestedLang);
        Matcher matcher = LANG_CONST_PATTERN.matcher(this.jsTemplate);
        String safe = TwHttpServer.escapeForJs(lang);
        String replaced = matcher.find() ? matcher.replaceFirst("const TWB_DEFAULT_LANG = \"" + safe + "\";") : this.jsTemplate;
        byte[] bytes = replaced.getBytes(StandardCharsets.UTF_8);
        return new JsVariant(bytes, TwHttpServer.calcEtag(bytes));
    }

    private String sanitizeLang(String rawLang) {
        if (rawLang == null) {
            return "en";
        }
        String normalized = rawLang.trim().replace('_', '-').toLowerCase(Locale.ROOT);
        if (normalized.isEmpty()) {
            return "en";
        }
        if (normalized.length() > 32) {
            return "en";
        }
        if (!LANG_SANITIZE_PATTERN.matcher(normalized).matches()) {
            return "en";
        }
        return normalized;
    }

    private String extractLang(URI uri) {
        if (uri == null) {
            return "en";
        }
        String query = uri.getRawQuery();
        if (query == null || query.isEmpty()) {
            return "en";
        }
        for (String part : query.split("&")) {
            String key;
            if (part.isEmpty()) continue;
            int eq = part.indexOf(61);
            String string = key = eq >= 0 ? part.substring(0, eq) : part;
            if (!"lang".equalsIgnoreCase(this.decodeSafe(key))) continue;
            String value = eq >= 0 ? part.substring(eq + 1) : "";
            return this.sanitizeLang(this.decodeSafe(value));
        }
        return "en";
    }

    private String decodeSafe(String value) {
        try {
            return URLDecoder.decode(value, StandardCharsets.UTF_8);
        }
        catch (Exception e) {
            return value;
        }
    }

    private static String escapeForJs(String input) {
        return input.replace("\\", "\\\\").replace("\"", "\\\"");
    }

    private String buildBlockListJson() {
        StringBuilder builder = new StringBuilder();
        builder.append("[");
        List<TwBridgePlugin.BlockEntry> blocks = this.plugin.getAvailableBlocks();
        for (int i = 0; i < blocks.size(); ++i) {
            TwBridgePlugin.BlockEntry b = blocks.get(i);
            if (i > 0) {
                builder.append(",");
            }
            builder.append("[\"").append(TwHttpServer.escapeForJs(b.name())).append("\",\"").append(TwHttpServer.escapeForJs(b.id())).append("\"]");
        }
        builder.append("]");
        return builder.toString();
    }

    private record JsVariant(byte[] bytes, String etag) {
    }
}

