/*
 * Decompiled with CFR 0.152.
 */
package net.nando256.twbridge.ws;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.security.SecureRandom;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.nando256.twbridge.TwBridgePlugin;
import org.java_websocket.WebSocket;
import org.java_websocket.handshake.ClientHandshake;
import org.java_websocket.server.WebSocketServer;
import org.json.JSONArray;
import org.json.JSONObject;

public class BridgeServer
extends WebSocketServer {
    private final TwBridgePlugin plugin;
    private final Map<WebSocket, Integer> counters = new ConcurrentHashMap<WebSocket, Integer>();
    private final Map<WebSocket, Session> sessions = new ConcurrentHashMap<WebSocket, Session>();
    private final Map<String, WebSocket> playerBindings = new ConcurrentHashMap<String, WebSocket>();
    private final Timer timer = new Timer(true);
    private final SecureRandom rng = new SecureRandom();
    private final boolean pairingRequired;
    private final int pairWindowSeconds;
    private final int maxMsgPerSec;
    private final int maxMsgBytes;
    private final Set<String> allowedOrigins;
    private volatile String activePairCode = null;
    private volatile long pairExpireAt = 0L;

    public BridgeServer(TwBridgePlugin plugin, String host, int port, Set<String> allowedOrigins, int maxMsgPerSec, int maxMsgBytes, boolean pairingRequired, int pairWindowSeconds) {
        super(new InetSocketAddress(host, port));
        this.plugin = plugin;
        this.allowedOrigins = allowedOrigins;
        this.maxMsgPerSec = maxMsgPerSec;
        this.maxMsgBytes = maxMsgBytes;
        this.pairingRequired = pairingRequired;
        this.pairWindowSeconds = pairWindowSeconds;
        this.timer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                BridgeServer.this.counters.replaceAll((k, v) -> 0);
            }
        }, 1000L, 1000L);
        if (pairingRequired) {
            this.rotatePairCode();
        }
    }

    public String rotatePairCode() {
        if (!this.pairingRequired) {
            return null;
        }
        this.activePairCode = String.format("%06d", this.rng.nextInt(1000000));
        this.pairExpireAt = System.currentTimeMillis() + (long)this.pairWindowSeconds * 1000L;
        this.plugin.getLogger().info("[twbridge] Pairing code: " + this.activePairCode + " (valid " + this.pairWindowSeconds + "s)");
        return this.activePairCode;
    }

    @Override
    public void onOpen(WebSocket conn, ClientHandshake hs) {
        InetAddress addr = conn.getRemoteSocketAddress().getAddress();
        if (addr == null) {
            conn.close(1008, "address unknown");
            return;
        }
        if (!this.isOriginAllowed(hs.getFieldValue("Origin"))) {
            conn.close(1008, "origin not allowed");
            return;
        }
        this.plugin.getLogger().info("[twbridge] WS connected: " + String.valueOf(conn.getRemoteSocketAddress()));
        this.plugin.logDebug("Connection opened: " + String.valueOf(conn.getRemoteSocketAddress()));
        this.counters.put(conn, 0);
        this.sendJson(conn, new JSONObject().put("hello", "twbridge").put("pairing", this.pairingRequired));
    }

    @Override
    public void onMessage(WebSocket conn, String message) {
        if (message.length() > this.maxMsgBytes) {
            conn.close(1009, "msg too large");
            return;
        }
        this.counters.compute(conn, (k, v) -> v == null ? 1 : v + 1);
        if (this.counters.get(conn) > this.maxMsgPerSec) {
            conn.close(1011, "rate limit");
            return;
        }
        try {
            JSONObject json = new JSONObject(message);
            UUID id = UUID.fromString(json.optString("id", UUID.randomUUID().toString()));
            String cmd = json.optString("cmd", "");
            if ("pair.start".equals(cmd)) {
                if (this.sessions.containsKey(conn)) {
                    this.err(conn, id, "session already established");
                    return;
                }
                String requestedPlayer = json.optString("player", "").trim();
                if (requestedPlayer.isEmpty()) {
                    this.err(conn, id, "player required");
                    conn.close(1008, "player required");
                    return;
                }
                String resolvedPlayer = this.plugin.resolveOnlinePlayerName(requestedPlayer);
                if (resolvedPlayer == null) {
                    this.err(conn, id, "player not online");
                    conn.close(1008, "player not online");
                    return;
                }
                if (!this.pairingRequired) {
                    String sessId = UUID.randomUUID().toString();
                    if (!this.bindPlayer(resolvedPlayer, conn)) {
                        this.err(conn, id, "player already bound");
                        conn.close(1008, "player already bound");
                        return;
                    }
                    this.sessions.put(conn, new Session(sessId, System.currentTimeMillis(), resolvedPlayer));
                    this.plugin.logDebug("Session established for " + String.valueOf(conn.getRemoteSocketAddress()) + " player=" + resolvedPlayer);
                    this.ok(conn, id, new JSONObject().put("sessionId", sessId));
                    return;
                }
                String code = json.optString("code", "");
                long now = System.currentTimeMillis();
                if (!code.equals(this.activePairCode) || now > this.pairExpireAt) {
                    this.err(conn, id, "invalid or expired code");
                    conn.close(1008, "invalid or expired code");
                    return;
                }
                String sessId = UUID.randomUUID().toString();
                if (!this.bindPlayer(resolvedPlayer, conn)) {
                    this.err(conn, id, "player already bound");
                    conn.close(1008, "player already bound");
                    return;
                }
                this.sessions.put(conn, new Session(sessId, now, resolvedPlayer));
                this.activePairCode = null;
                this.pairExpireAt = 0L;
                this.ok(conn, id, new JSONObject().put("sessionId", sessId));
                this.plugin.logDebug("Session established for " + String.valueOf(conn.getRemoteSocketAddress()) + " player=" + resolvedPlayer);
                return;
            }
            if (this.pairingRequired && !this.requireActiveSession(conn, json)) {
                this.err(conn, id, "not paired");
                conn.close(1008, "pairing required");
                return;
            }
            if ("command.run".equals(cmd)) {
                String command = json.optString("command", "").trim();
                if (command.isEmpty()) {
                    this.err(conn, id, "command missing");
                    return;
                }
                this.plugin.logDebug("command.run: " + command);
                this.plugin.handleCommand(command, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "command failed" : msg));
                return;
            }
            if ("agent.teleportToPlayer".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.teleportToPlayer id=" + agentId + " player=" + owner);
                this.plugin.handleAgentTeleportToPlayer(agentId, owner, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "teleport failed" : msg));
                return;
            }
            if ("agent.move".equals(cmd)) {
                String owner;
                double blocks;
                String agentId = json.optString("agentId", "").trim();
                String direction = json.optString("direction", "forward").trim();
                double d = blocks = json.has("blocks") ? json.optDouble("blocks", Double.NaN) : 0.0;
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                if (!Double.isFinite(blocks)) {
                    this.err(conn, id, "blocks must be a number");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.move id=" + agentId + " player=" + owner + " dir=" + direction + " blocks=" + blocks);
                this.plugin.handleAgentMove(agentId, owner, direction, blocks, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "move failed" : msg));
                return;
            }
            if ("agent.rotate".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                String direction = json.optString("direction", "left").trim();
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.rotate id=" + agentId + " player=" + owner + " dir=" + direction);
                this.plugin.handleAgentRotate(agentId, owner, direction, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "rotate failed" : msg));
                return;
            }
            if ("agent.slotActivate".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                int slot = json.optInt("slot", -1);
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                if (slot < 1 || slot > 27) {
                    this.err(conn, id, "slot must be 1-27");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.slotActivate id=" + agentId + " player=" + owner + " slot=" + slot);
                this.plugin.handleAgentSlotActivate(agentId, owner, slot, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "slot activate failed" : msg));
                return;
            }
            if ("blocks.list".equals(cmd)) {
                JSONArray array = new JSONArray();
                this.plugin.getAvailableBlocks().forEach(block -> array.put(new JSONObject().put("id", block.id()).put("name", block.name())));
                this.ok(conn, id, new JSONObject().put("blocks", array));
                return;
            }
            if ("agent.place".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                String direction = json.optString("direction", "forward").trim();
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.place id=" + agentId + " player=" + owner + " dir=" + direction);
                this.plugin.handleAgentPlace(agentId, owner, direction, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "place failed" : msg));
                return;
            }
            if ("agent.slotSetBlock".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                String block2 = json.optString("block", "").trim();
                int amount = json.optInt("amount", -1);
                int slot = json.optInt("slot", -1);
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                if (slot < 1 || slot > 27) {
                    this.err(conn, id, "slot must be 1-27");
                    return;
                }
                if (amount < 1 || amount > 64) {
                    this.err(conn, id, "amount must be 1-64");
                    return;
                }
                if (block2.isEmpty()) {
                    this.err(conn, id, "block required");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.slotSetBlock id=" + agentId + " player=" + owner + " slot=" + slot + " block=" + block2 + " amount=" + amount);
                this.plugin.handleAgentSlotAssignBlock(agentId, owner, block2, amount, slot, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "slot set failed" : msg));
                return;
            }
            if ("agent.despawn".equals(cmd)) {
                String owner;
                String agentId = json.optString("agentId", "").trim();
                if (agentId.isEmpty()) {
                    this.err(conn, id, "agentId required");
                    return;
                }
                Session session = this.sessions.get(conn);
                String string = owner = session == null ? null : session.player();
                if (owner == null || owner.isBlank()) {
                    this.err(conn, id, "player not bound");
                    return;
                }
                this.plugin.logDebug("agent.despawn id=" + agentId + " player=" + owner);
                this.plugin.handleAgentDespawn(agentId, owner, () -> this.ok(conn, id, null), msg -> this.err(conn, id, msg == null ? "despawn failed" : msg));
                return;
            }
            this.err(conn, id, "unknown cmd: " + cmd);
        }
        catch (Exception e) {
            conn.close(1011, "bad message");
        }
    }

    private boolean requireActiveSession(WebSocket conn, JSONObject json) {
        if (!this.pairingRequired) {
            return true;
        }
        Session session = this.sessions.get(conn);
        return session != null && json.optString("sessionId", "").equals(session.sessionId());
    }

    private void sendJson(WebSocket conn, JSONObject obj) {
        conn.send(obj.toString());
    }

    private void ok(WebSocket conn, UUID id, JSONObject res) {
        JSONObject payload = new JSONObject().put("id", id.toString()).put("ok", true);
        if (res != null) {
            payload.put("result", res);
        }
        this.sendJson(conn, payload);
    }

    private void err(WebSocket conn, UUID id, String msg) {
        this.sendJson(conn, new JSONObject().put("id", id.toString()).put("ok", false).put("error", msg));
    }

    private boolean isOriginAllowed(String origin) {
        if (origin == null || origin.isBlank()) {
            return true;
        }
        if (this.allowedOrigins.isEmpty()) {
            return true;
        }
        if (this.allowedOrigins.contains("*")) {
            return true;
        }
        return this.allowedOrigins.contains(origin);
    }

    @Override
    public void onClose(WebSocket conn, int code, String reason, boolean remote) {
        this.plugin.getLogger().info("[twbridge] WS disconnected: " + String.valueOf(conn.getRemoteSocketAddress()) + " code=" + code + " reason=" + reason);
        this.counters.remove(conn);
        Session session = this.sessions.remove(conn);
        if (session != null && session.player() != null) {
            String normalized = session.player().toLowerCase(Locale.ROOT);
            this.playerBindings.remove(normalized, conn);
        }
    }

    @Override
    public void onError(WebSocket conn, Exception ex) {
        String message = ex == null ? "unknown" : ex.getMessage();
        this.plugin.getLogger().warning("[twbridge] WS error: " + message);
    }

    @Override
    public void onStart() {
        this.plugin.getLogger().info("[twbridge] BridgeServer listening on " + String.valueOf(this.getAddress()));
    }

    private boolean bindPlayer(String playerName, WebSocket conn) {
        String normalized = playerName.toLowerCase(Locale.ROOT);
        WebSocket existing = this.playerBindings.putIfAbsent(normalized, conn);
        return existing == null || existing == conn;
    }

    private record Session(String sessionId, long createdAt, String player) {
    }
}

