/*
 * Decompiled with CFR 0.152.
 */
package net.nando256.pdca;

import io.papermc.paper.event.player.PlayerLecternPageChangeEvent;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.nando256.pdca.PdcaTimerPlugin;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.Sound;
import org.bukkit.World;
import org.bukkit.block.Block;
import org.bukkit.block.BlockState;
import org.bukkit.block.Lectern;
import org.bukkit.boss.BarColor;
import org.bukkit.boss.BarFlag;
import org.bukkit.boss.BarStyle;
import org.bukkit.boss.BossBar;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.entity.Player;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.Listener;
import org.bukkit.event.block.BlockBreakEvent;
import org.bukkit.event.player.PlayerInteractEvent;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.meta.BookMeta;
import org.bukkit.inventory.meta.ItemMeta;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitScheduler;
import org.bukkit.scheduler.BukkitTask;
import org.jetbrains.annotations.Nullable;

final class PdcaTimerManager
implements Listener {
    private static final Pattern STEP_PATTERN = Pattern.compile("^\\[(\\d+)]\\s*(.*)$");
    private static final Pattern DURATION_PATTERN = Pattern.compile("^(\\d+)([hms])?$");
    private static final Pattern OPTION_PATTERN = Pattern.compile("^\\[option\\s+([^\\]]+)]$", 2);
    private static final BarColor[] STEP_BAR_COLORS = new BarColor[]{BarColor.RED, BarColor.BLUE, BarColor.GREEN, BarColor.YELLOW, BarColor.PURPLE, BarColor.WHITE, BarColor.PINK};
    private static final int PROGRESS_SLOTS = 180;
    private static final String[] STEP_COLOR_CODES = new String[]{"\u00a7c", "\u00a79", "\u00a7a", "\u00a7e", "\u00a75", "\u00a7f", "\u00a7b"};
    private final PdcaTimerPlugin plugin;
    private final Map<LecternKey, Session> sessions = new HashMap<LecternKey, Session>();
    private final double notifyRadius;
    private final Duration warnBefore;
    private final int titleFadeInTicks;
    private final int titleStayTicks;
    private final int titleFadeOutTicks;
    private final Sound startSound;
    private final Sound warnSound;
    private final Sound endSound;

    PdcaTimerManager(PdcaTimerPlugin plugin) {
        this.plugin = plugin;
        ConfigurationSection cfg = plugin.getConfig().getConfigurationSection("pdca");
        if (cfg == null) {
            cfg = plugin.getConfig().createSection("pdca");
        }
        this.notifyRadius = cfg.getDouble("radius", 10.0);
        this.warnBefore = this.parseDuration(cfg.getString("warn_before"), Duration.ofMinutes(5L));
        ConfigurationSection title = cfg.getConfigurationSection("title");
        this.titleFadeInTicks = title != null ? title.getInt("fade_in_ticks", 10) : 10;
        this.titleStayTicks = title != null ? title.getInt("stay_ticks", 60) : 60;
        this.titleFadeOutTicks = title != null ? title.getInt("fade_out_ticks", 10) : 10;
        ConfigurationSection sounds = cfg.getConfigurationSection("sounds");
        this.startSound = this.parseSound(sounds != null ? sounds.getString("start") : null, Sound.ENTITY_EXPERIENCE_ORB_PICKUP);
        this.warnSound = this.parseSound(sounds != null ? sounds.getString("warn") : null, Sound.BLOCK_NOTE_BLOCK_BELL);
        this.endSound = this.parseSound(sounds != null ? sounds.getString("end") : null, Sound.UI_TOAST_CHALLENGE_COMPLETE);
    }

    private Sound parseSound(@Nullable String raw, Sound fallback) {
        if (raw == null || raw.isBlank()) {
            return fallback;
        }
        try {
            return Sound.valueOf((String)raw.trim().toUpperCase(Locale.ROOT));
        }
        catch (IllegalArgumentException ignored) {
            this.plugin.getLogger().warning("[PDCATimer] Invalid sound '" + raw + "'. Falling back to " + String.valueOf(fallback));
            return fallback;
        }
    }

    void shutdown() {
        for (Session session : new ArrayList<Session>(this.sessions.values())) {
            session.cancel(false);
        }
        this.sessions.clear();
    }

    @EventHandler(ignoreCancelled=true, priority=EventPriority.MONITOR)
    public void onRightClickLectern(PlayerInteractEvent event) {
        Block clicked = event.getClickedBlock();
        if (clicked == null || clicked.getType() != Material.LECTERN) {
            return;
        }
        this.plugin.getServer().getScheduler().runTask((Plugin)this.plugin, () -> this.tryStartFromLectern(clicked, event.getPlayer()));
    }

    @EventHandler(ignoreCancelled=true)
    public void onPageChange(PlayerLecternPageChangeEvent event) {
        this.tryStartFromLectern(event.getLectern().getBlock(), event.getPlayer());
    }

    @EventHandler(ignoreCancelled=true)
    public void onLecternBreak(BlockBreakEvent event) {
        if (event.getBlock().getType() != Material.LECTERN) {
            return;
        }
        this.cancelSession(LecternKey.fromBlock(event.getBlock()), true);
    }

    private void tryStartFromLectern(Block block, Player starter) {
        String normalized;
        BlockState state = block.getState();
        if (!(state instanceof Lectern)) {
            return;
        }
        Lectern lectern = (Lectern)state;
        ItemStack bookStack = lectern.getInventory().getItem(0);
        if (bookStack == null) {
            return;
        }
        Material type = bookStack.getType();
        if (type != Material.WRITTEN_BOOK && type != Material.WRITABLE_BOOK) {
            return;
        }
        ItemMeta itemMeta = bookStack.getItemMeta();
        if (!(itemMeta instanceof BookMeta)) {
            return;
        }
        BookMeta meta = (BookMeta)itemMeta;
        String firstPage = "";
        if (meta.getPageCount() > 0) {
            try {
                firstPage = meta.getPage(1);
            }
            catch (IndexOutOfBoundsException ignored) {
                firstPage = "";
            }
        }
        String string = normalized = firstPage == null ? "" : firstPage.strip();
        if (!normalized.toUpperCase(Locale.ROOT).startsWith("[PDCA]")) {
            return;
        }
        ParsedBook parsed = this.parseBook(firstPage);
        if (parsed.steps().isEmpty()) {
            starter.sendMessage("\u00a7c[PDCA] No steps were found on page 1 of the book.");
            return;
        }
        LecternKey key = LecternKey.fromBlock(block);
        if (this.sessions.containsKey(key)) {
            starter.sendMessage("\u00a7e[PDCA] A PDCA timer is already running for this lectern.");
            return;
        }
        Location center = block.getLocation().toCenterLocation().add(0.0, 0.25, 0.0);
        Session session = new Session(key, center, parsed.steps(), parsed.totalDuration(), parsed.displayMode(), parsed.audienceMode(), parsed.audienceNames(), starter.getUniqueId(), starter.getName());
        this.sessions.put(key, session);
        session.start();
        starter.sendMessage("\u00a7a[PDCA] Started PDCA timer (steps: " + parsed.steps().size() + ")");
    }

    private void cancelSession(LecternKey key, boolean removed) {
        Session session = this.sessions.remove(key);
        if (session != null) {
            session.cancel(removed);
        }
    }

    private ParsedBook parseBook(String firstPage) {
        String[] rawLines = firstPage.replace("\r", "").split("\n");
        ArrayList<String> stepLines = new ArrayList<String>();
        DisplayMode displayMode = DisplayMode.OVERALL_ONLY;
        AudienceMode audienceMode = AudienceMode.NEARBY;
        HashSet<String> audienceNames = new HashSet<String>();
        for (String raw : rawLines) {
            String[] tokens;
            String value;
            String line = raw.trim();
            if (line.isEmpty()) continue;
            Matcher optionMatcher = OPTION_PATTERN.matcher(line);
            if (!optionMatcher.matches()) {
                stepLines.add(line);
                continue;
            }
            String option = optionMatcher.group(1).toLowerCase(Locale.ROOT);
            if (option.contains("display")) {
                value = option.replace("display", "").replace('=', ' ').trim();
                tokens = value.isEmpty() ? new String[]{} : value.split("\\s+");
                boolean hasOverall = false;
                boolean hasStep = false;
                boolean hasNone = false;
                boolean hasBoth = false;
                for (String token : tokens) {
                    if (token.isBlank()) continue;
                    if (token.contains("overall")) {
                        hasOverall = true;
                    }
                    if (token.contains("step")) {
                        hasStep = true;
                    }
                    if (token.equals("none") || token.equals("hide") || token.equals("off")) {
                        hasNone = true;
                    }
                    if (!token.equals("both") && !token.equals("all")) continue;
                    hasBoth = true;
                }
                if (hasNone) {
                    displayMode = DisplayMode.NONE;
                    continue;
                }
                if (hasBoth || hasOverall && hasStep) {
                    displayMode = DisplayMode.BOTH;
                    continue;
                }
                if (hasOverall) {
                    displayMode = DisplayMode.OVERALL_ONLY;
                    continue;
                }
                if (!hasStep) continue;
                displayMode = DisplayMode.PER_STEP_ONLY;
                continue;
            }
            if (option.contains("audience") || option.contains("user")) {
                value = option.replace("audience", "").replace("user", "").replace('=', ' ').replace(',', ' ').trim();
                tokens = value.isEmpty() ? new String[]{} : value.split("\\s+");
                boolean anyToken = false;
                block21: for (String token : tokens) {
                    if (token.isBlank()) continue;
                    anyToken = true;
                    switch (token) {
                        case "nearby": 
                        case "radius": {
                            audienceMode = AudienceMode.NEARBY;
                            continue block21;
                        }
                        case "all": 
                        case "everyone": 
                        case "global": {
                            audienceMode = AudienceMode.ALL;
                            continue block21;
                        }
                        case "none": 
                        case "off": 
                        case "hide": {
                            audienceMode = AudienceMode.NONE;
                            continue block21;
                        }
                        case "list": 
                        case "users": 
                        case "players": {
                            audienceMode = AudienceMode.LIST;
                            continue block21;
                        }
                        default: {
                            audienceNames.add(token.toLowerCase(Locale.ROOT));
                            audienceMode = AudienceMode.LIST;
                        }
                    }
                }
                if (anyToken) continue;
                audienceMode = AudienceMode.NEARBY;
                continue;
            }
            stepLines.add(line);
        }
        if (audienceMode == AudienceMode.LIST && audienceNames.isEmpty()) {
            audienceMode = AudienceMode.NEARBY;
        }
        List<PdcaStep> steps = this.parseSteps(stepLines);
        Duration total = Duration.ZERO;
        for (PdcaStep step : steps) {
            total = total.plus(step.duration());
        }
        return new ParsedBook(steps, total, displayMode, audienceMode, Collections.unmodifiableSet(audienceNames));
    }

    private List<PdcaStep> parseSteps(List<String> lines) {
        LinkedHashMap<Integer, PdcaStep> map = new LinkedHashMap<Integer, PdcaStep>();
        for (String raw : lines) {
            Duration duration;
            Object label;
            String durationToken;
            int id;
            Matcher matcher;
            String line = raw.trim();
            if (line.isEmpty() || line.equalsIgnoreCase("[PDCA]") || !(matcher = STEP_PATTERN.matcher(line)).matches()) continue;
            try {
                id = Integer.parseInt(matcher.group(1));
            }
            catch (NumberFormatException ex) {
                continue;
            }
            String remainder = matcher.group(2).trim();
            int colon = remainder.indexOf(58);
            if (colon >= 0) {
                durationToken = remainder.substring(0, colon).trim();
                label = remainder.substring(colon + 1).trim();
            } else {
                durationToken = remainder;
                label = "";
            }
            if (durationToken.isEmpty() || (duration = this.parseDuration(durationToken = durationToken.replace(" ", ""))) == null || duration.isZero() || duration.isNegative()) continue;
            if (((String)label).isEmpty()) {
                label = "Step " + id;
            }
            map.put(id, new PdcaStep(id, duration, (String)label));
        }
        ArrayList<PdcaStep> steps = new ArrayList<PdcaStep>(map.values());
        steps.sort(Comparator.comparingInt(PdcaStep::id));
        return steps;
    }

    private Duration parseDuration(String token, Duration fallback) {
        Duration parsed = this.parseDuration(token);
        return parsed != null ? parsed : fallback;
    }

    @Nullable
    private Duration parseDuration(String token) {
        if (token == null) {
            return null;
        }
        String trimmed = token.trim().toLowerCase(Locale.ROOT);
        Matcher matcher = DURATION_PATTERN.matcher(trimmed);
        if (!matcher.matches()) {
            return null;
        }
        long value = Long.parseLong(matcher.group(1));
        String suffix = matcher.group(2);
        if (suffix == null || suffix.equals("m")) {
            return Duration.ofMinutes(value);
        }
        return switch (suffix) {
            case "s" -> Duration.ofSeconds(value);
            case "h" -> Duration.ofHours(value);
            default -> null;
        };
    }

    private List<Player> audience(Location center) {
        if (this.notifyRadius <= 0.0) {
            return Collections.emptyList();
        }
        double radiusSq = this.notifyRadius * this.notifyRadius;
        ArrayList<Player> players = new ArrayList<Player>();
        for (Player player : Objects.requireNonNull(center.getWorld()).getPlayers()) {
            if (!(player.getLocation().distanceSquared(center) <= radiusSq)) continue;
            players.add(player);
        }
        return players;
    }

    private boolean hasPdcaBook(LecternKey key) {
        Block block = key.block();
        if (block == null || block.getType() != Material.LECTERN) {
            return false;
        }
        Lectern lectern = (Lectern)block.getState();
        ItemStack item = lectern.getInventory().getItem(0);
        if (item == null) {
            return false;
        }
        Material type = item.getType();
        if (type != Material.WRITTEN_BOOK && type != Material.WRITABLE_BOOK) {
            return false;
        }
        ItemMeta itemMeta = item.getItemMeta();
        if (!(itemMeta instanceof BookMeta)) {
            return false;
        }
        BookMeta meta = (BookMeta)itemMeta;
        if (meta.getPageCount() == 0) {
            return false;
        }
        String first = meta.getPage(1);
        return first != null && first.strip().toUpperCase(Locale.ROOT).startsWith("[PDCA]");
    }

    private static Duration clampToZero(Duration duration) {
        return duration.isNegative() ? Duration.ZERO : duration;
    }

    private static String formatDuration(Duration duration) {
        duration = PdcaTimerManager.clampToZero(duration);
        long seconds = duration.getSeconds();
        long hours = seconds / 3600L;
        long minutes = seconds % 3600L / 60L;
        long secs = seconds % 60L;
        if (hours > 0L) {
            return String.format("%d:%02d:%02d", hours, minutes, secs);
        }
        return String.format("%d:%02d", minutes, secs);
    }

    private final class Session
    implements Runnable {
        private final LecternKey key;
        private final Location center;
        private final List<PdcaStep> steps;
        private final Duration totalDuration;
        private final long totalDurationMillis;
        private final DisplayMode displayMode;
        private final AudienceMode audienceMode;
        private final Set<String> audienceNames;
        private final long[] stepDurationsMillis;
        private final long[] stepStartOffsetsMillis;
        private final int[] stepSlotWidths;
        private final UUID starter;
        private final String starterName;
        private int index = 0;
        private PdcaStep currentStep;
        private long sessionStartMillis;
        private long stepStartMillis;
        private BossBar overallBar;
        private BossBar countdownBar;
        private final Set<UUID> viewers = new HashSet<UUID>();
        private BukkitTask stepTask;
        private BukkitTask warnTask;
        private BukkitTask progressTask;

        Session(LecternKey key, Location center, List<PdcaStep> steps, Duration totalDuration, DisplayMode displayMode, AudienceMode audienceMode, Set<String> audienceNames, UUID starter, String starterName) {
            this.key = key;
            this.center = center;
            this.steps = steps;
            this.totalDuration = totalDuration;
            this.totalDurationMillis = Math.max(1L, totalDuration.toMillis());
            this.stepDurationsMillis = new long[steps.size()];
            this.stepStartOffsetsMillis = new long[steps.size()];
            long offset = 0L;
            for (int i = 0; i < steps.size(); ++i) {
                long millis;
                this.stepStartOffsetsMillis[i] = offset;
                this.stepDurationsMillis[i] = millis = Math.max(1L, steps.get(i).duration().toMillis());
                offset += millis;
            }
            this.stepSlotWidths = new int[steps.size()];
            int slotsRemaining = 180;
            for (int i = 0; i < steps.size(); ++i) {
                int width;
                long durationMillis = this.stepDurationsMillis[i];
                int minRemaining = Math.max(0, steps.size() - i - 1);
                if (this.totalDurationMillis == 0L) {
                    width = Math.max(1, slotsRemaining / (steps.size() - i));
                } else if (i == steps.size() - 1) {
                    width = Math.max(1, slotsRemaining);
                } else {
                    double ratio = (double)durationMillis / (double)this.totalDurationMillis;
                    width = Math.max(1, (int)Math.round(ratio * 180.0));
                    if (width > slotsRemaining - minRemaining) {
                        width = slotsRemaining - minRemaining;
                    }
                    width = Math.max(1, width);
                }
                this.stepSlotWidths[i] = width;
                slotsRemaining -= width;
            }
            if (steps.size() > 0 && slotsRemaining != 0) {
                int n = steps.size() - 1;
                this.stepSlotWidths[n] = this.stepSlotWidths[n] + slotsRemaining;
            }
            this.displayMode = displayMode;
            this.audienceMode = audienceMode;
            this.audienceNames = audienceNames;
            this.starter = starter;
            this.starterName = starterName;
        }

        void start() {
            if (!PdcaTimerManager.this.hasPdcaBook(this.key)) {
                this.cancel(true);
                return;
            }
            this.sessionStartMillis = System.currentTimeMillis();
            if (this.showOverall()) {
                this.overallBar = Bukkit.createBossBar((String)"", (BarColor)BarColor.WHITE, (BarStyle)BarStyle.SOLID, (BarFlag[])new BarFlag[0]);
                this.overallBar.setProgress(0.0);
            } else {
                this.overallBar = null;
            }
            if (this.showPerStep()) {
                this.countdownBar = Bukkit.createBossBar((String)"PDCA", (BarColor)this.colorForIndex(0), (BarStyle)BarStyle.SEGMENTED_20, (BarFlag[])new BarFlag[0]);
                this.countdownBar.setProgress(1.0);
            } else {
                this.countdownBar = null;
            }
            Player starterPlayer = Bukkit.getPlayer((UUID)this.starter);
            if (starterPlayer != null) {
                this.addViewer(starterPlayer);
            }
            this.run();
            this.progressTask = PdcaTimerManager.this.plugin.getServer().getScheduler().runTaskTimer((Plugin)PdcaTimerManager.this.plugin, this::updateBossBarProgress, 0L, 10L);
        }

        @Override
        public void run() {
            if (!PdcaTimerManager.this.hasPdcaBook(this.key)) {
                this.cancel(true);
                return;
            }
            if (this.index >= this.steps.size()) {
                this.finish();
                return;
            }
            this.currentStep = this.steps.get(this.index);
            this.stepStartMillis = System.currentTimeMillis();
            if (this.countdownBar != null) {
                this.countdownBar.setColor(this.colorForIndex(this.index));
                this.countdownBar.setTitle(this.currentStep.label() + " remaining " + PdcaTimerManager.formatDuration(this.currentStep.duration()));
            }
            this.syncBossBarPlayers();
            for (Player player : this.messagePlayers()) {
                String label = this.currentStep.label();
                String mainTitle = label.contains(":") ? label.substring(label.indexOf(":") + 1).trim() : "";
                String subTitle = label.contains(":") ? label.substring(0, label.indexOf(":")).trim() : label;
                player.sendTitle(mainTitle, subTitle, PdcaTimerManager.this.titleFadeInTicks, PdcaTimerManager.this.titleStayTicks, PdcaTimerManager.this.titleFadeOutTicks);
                player.playSound(this.center, PdcaTimerManager.this.startSound, 1.0f, 1.0f);
            }
            this.warnTask = this.scheduleWarn(this.currentStep);
            this.stepTask = this.scheduleLater(() -> {
                ++this.index;
                this.run();
            }, this.currentStep.duration());
        }

        void finish() {
            this.cancelTasks();
            PdcaTimerManager.this.sessions.remove(this.key);
            if (!PdcaTimerManager.this.hasPdcaBook(this.key)) {
                this.messageStarter("\u00a7c[PDCA] Timer cancelled because the book was removed.");
                return;
            }
            for (Player player : this.messagePlayers()) {
                player.sendActionBar("\u00a7aTime for today's reflection!");
                player.playSound(this.center, PdcaTimerManager.this.endSound, 1.0f, 1.0f);
                player.playSound(player.getLocation(), Sound.BLOCK_BELL_USE, 1.0f, 1.0f);
            }
            this.messageStarter("\u00a7a[PDCA] Timer completed.");
        }

        void cancel(boolean removed) {
            this.cancelTasks();
            PdcaTimerManager.this.sessions.remove(this.key);
            if (removed) {
                this.messageStarter("\u00a7c[PDCA] Timer cancelled because the book was removed.");
            } else {
                this.messageStarter("\u00a7c[PDCA] Timer stopped.");
            }
        }

        private BukkitTask scheduleWarn(PdcaStep step) {
            if (PdcaTimerManager.this.warnBefore.isZero() || PdcaTimerManager.this.warnBefore.isNegative()) {
                return null;
            }
            if (step.duration().compareTo(PdcaTimerManager.this.warnBefore) <= 0) {
                return null;
            }
            Duration delay = step.duration().minus(PdcaTimerManager.this.warnBefore);
            return this.scheduleLater(() -> {
                if (!PdcaTimerManager.this.hasPdcaBook(this.key)) {
                    this.cancel(true);
                    return;
                }
                String warning = step.label() + " has " + PdcaTimerManager.formatDuration(PdcaTimerManager.this.warnBefore) + " remaining";
                for (Player player : this.messagePlayers()) {
                    player.sendActionBar(warning);
                    player.playSound(this.center, PdcaTimerManager.this.warnSound, 1.0f, 1.0f);
                }
            }, delay);
        }

        private BukkitTask scheduleLater(Runnable run, Duration duration) {
            long ticks = Math.max(1L, (duration.toMillis() + 49L) / 50L);
            BukkitScheduler scheduler = PdcaTimerManager.this.plugin.getServer().getScheduler();
            return scheduler.runTaskLater((Plugin)PdcaTimerManager.this.plugin, run, ticks);
        }

        private void updateBossBarProgress() {
            this.syncBossBarPlayers();
            long now = System.currentTimeMillis();
            long elapsed = now - this.sessionStartMillis;
            if (this.showOverall()) {
                this.broadcastOverallProgress(elapsed);
            }
            if (this.showPerStep() && this.countdownBar != null && this.currentStep != null) {
                long stepElapsed = now - this.stepStartMillis;
                long durationMillis = Math.max(1L, this.currentStep.duration().toMillis());
                double remaining = Math.max(0.0, 1.0 - (double)stepElapsed / (double)durationMillis);
                this.countdownBar.setProgress(Math.max(0.0, Math.min(1.0, remaining)));
                Duration remainingDuration = PdcaTimerManager.clampToZero(this.currentStep.duration().minusMillis(stepElapsed));
                this.countdownBar.setTitle(this.currentStep.label() + " remaining " + PdcaTimerManager.formatDuration(remainingDuration));
            }
        }

        private void broadcastOverallProgress(long elapsedMillis) {
            double progress;
            if (this.totalDurationMillis <= 0L || this.overallBar == null) {
                return;
            }
            long clamped = Math.max(0L, Math.min(this.totalDurationMillis, elapsedMillis));
            StringBuilder bar = new StringBuilder("\u00a77[");
            for (int i = 0; i < this.steps.size(); ++i) {
                int width = this.stepSlotWidths[i];
                long stepStart = this.stepStartOffsetsMillis[i];
                long stepDuration = this.stepDurationsMillis[i];
                double stepProgress = clamped <= stepStart ? 0.0 : (clamped >= stepStart + stepDuration ? 1.0 : (double)(clamped - stepStart) / (double)stepDuration);
                int filled = Math.min(width, Math.max(0, (int)Math.round(stepProgress * (double)width)));
                String color = STEP_COLOR_CODES[i % STEP_COLOR_CODES.length];
                for (int slot = 0; slot < width; ++slot) {
                    boolean boundary;
                    boolean bl = boundary = slot == 0;
                    if (slot < filled || boundary) {
                        bar.append(color).append('|');
                        continue;
                    }
                    bar.append("\u00a77|");
                }
                bar.append("\u00a77|");
            }
            bar.append("\u00a77]");
            Duration remaining = PdcaTimerManager.clampToZero(this.totalDuration.minusMillis(clamped));
            double d = progress = this.totalDurationMillis == 0L ? 0.0 : (double)clamped / (double)this.totalDurationMillis;
            if (this.overallBar != null) {
                this.overallBar.setProgress(Math.max(0.0, Math.min(1.0, progress)));
                this.overallBar.setTitle(String.valueOf(bar) + PdcaTimerManager.formatDuration(remaining));
            }
        }

        private void syncBossBarPlayers() {
            List<Player> current = this.audiencePlayers();
            HashSet<UUID> active = new HashSet<UUID>();
            for (Player player : current) {
                active.add(player.getUniqueId());
                this.addViewer(player);
            }
            this.viewers.removeIf(uuid -> {
                boolean allowed;
                Player player = Bukkit.getPlayer((UUID)uuid);
                if (player == null || !player.isOnline()) {
                    this.removeViewer(player);
                    return true;
                }
                AudienceMode mode = this.audienceMode;
                switch (mode.ordinal()) {
                    default: {
                        throw new MatchException(null, null);
                    }
                    case 1: {
                        boolean bl = true;
                        break;
                    }
                    case 2: {
                        boolean bl = this.audienceNames.contains(player.getName().toLowerCase(Locale.ROOT));
                        break;
                    }
                    case 3: {
                        boolean bl = false;
                        break;
                    }
                    case 0: {
                        boolean bl = allowed = true;
                    }
                }
                if (!allowed) {
                    this.removeViewer(player);
                    return true;
                }
                if (!(mode != AudienceMode.ALL && mode != AudienceMode.LIST || active.contains(uuid))) {
                    this.removeViewer(player);
                    return true;
                }
                this.ensureViewer(player);
                return false;
            });
        }

        private List<Player> audiencePlayers() {
            return switch (this.audienceMode.ordinal()) {
                case 1 -> new ArrayList(PdcaTimerManager.this.plugin.getServer().getOnlinePlayers());
                case 2 -> {
                    ArrayList<Player> players = new ArrayList<Player>();
                    for (Player player : PdcaTimerManager.this.plugin.getServer().getOnlinePlayers()) {
                        if (!this.audienceNames.contains(player.getName().toLowerCase(Locale.ROOT))) continue;
                        players.add(player);
                    }
                    yield players;
                }
                case 3 -> Collections.emptyList();
                default -> PdcaTimerManager.this.audience(this.center);
            };
        }

        private List<Player> messagePlayers() {
            ArrayList<Player> players = new ArrayList<Player>();
            for (UUID uuid : this.viewers) {
                Player player = Bukkit.getPlayer((UUID)uuid);
                if (player == null || !player.isOnline() || !this.isPlayerAllowed(player)) continue;
                players.add(player);
            }
            return players;
        }

        private List<Player> countdownPlayers() {
            if (!this.showPerStep()) {
                return Collections.emptyList();
            }
            ArrayList<Player> players = new ArrayList<Player>();
            for (UUID uuid : this.viewers) {
                Player player = Bukkit.getPlayer((UUID)uuid);
                if (player == null || !player.isOnline() || !this.isPlayerAllowed(player)) continue;
                players.add(player);
            }
            return players;
        }

        private void cancelTasks() {
            if (this.stepTask != null) {
                this.stepTask.cancel();
                this.stepTask = null;
            }
            if (this.warnTask != null) {
                this.warnTask.cancel();
                this.warnTask = null;
            }
            if (this.progressTask != null) {
                this.progressTask.cancel();
                this.progressTask = null;
            }
            if (this.overallBar != null) {
                this.overallBar.removeAll();
                this.overallBar = null;
            }
            if (this.countdownBar != null) {
                this.countdownBar.removeAll();
                this.countdownBar = null;
            }
            this.viewers.clear();
        }

        private void messageStarter(String message) {
            Optional.ofNullable(Bukkit.getPlayer((UUID)this.starter)).ifPresentOrElse(player -> player.sendMessage(message), () -> PdcaTimerManager.this.plugin.getLogger().info(message + " (" + this.starterName + ")"));
        }

        private void addViewer(Player player) {
            if (player == null || !this.isPlayerAllowed(player)) {
                return;
            }
            if (this.viewers.add(player.getUniqueId())) {
                if (this.overallBar != null) {
                    this.overallBar.addPlayer(player);
                }
                if (this.countdownBar != null) {
                    this.countdownBar.addPlayer(player);
                }
            } else {
                this.ensureViewer(player);
            }
        }

        private void ensureViewer(Player player) {
            if (player == null || !this.isPlayerAllowed(player)) {
                return;
            }
            if (this.overallBar != null && !this.overallBar.getPlayers().contains(player)) {
                this.overallBar.addPlayer(player);
            }
            if (this.countdownBar != null && !this.countdownBar.getPlayers().contains(player)) {
                this.countdownBar.addPlayer(player);
            }
        }

        private void removeViewer(@Nullable Player player) {
            if (player == null) {
                return;
            }
            if (this.overallBar != null) {
                this.overallBar.removePlayer(player);
            }
            if (this.countdownBar != null) {
                this.countdownBar.removePlayer(player);
            }
        }

        private boolean isPlayerAllowed(Player player) {
            if (player == null) {
                return false;
            }
            return switch (this.audienceMode.ordinal()) {
                default -> throw new MatchException(null, null);
                case 1 -> true;
                case 2 -> this.audienceNames.contains(player.getName().toLowerCase(Locale.ROOT));
                case 3 -> false;
                case 0 -> true;
            };
        }

        private boolean showOverall() {
            return this.displayMode == DisplayMode.BOTH || this.displayMode == DisplayMode.OVERALL_ONLY;
        }

        private boolean showPerStep() {
            return this.displayMode == DisplayMode.BOTH || this.displayMode == DisplayMode.PER_STEP_ONLY;
        }

        private BarColor colorForIndex(int idx) {
            return STEP_BAR_COLORS[idx % STEP_BAR_COLORS.length];
        }

        private int stepIndexForFraction(double fraction) {
            double targetMillis = fraction * (double)this.totalDurationMillis;
            for (int i = 0; i < this.stepStartOffsetsMillis.length; ++i) {
                long start = this.stepStartOffsetsMillis[i];
                long end = start + this.stepDurationsMillis[i];
                if (!(targetMillis <= (double)end) && i != this.stepStartOffsetsMillis.length - 1) continue;
                return i;
            }
            return Math.max(0, this.stepStartOffsetsMillis.length - 1);
        }
    }

    private record LecternKey(String world, int x, int y, int z) {
        static LecternKey fromBlock(Block block) {
            return new LecternKey(block.getWorld().getName(), block.getX(), block.getY(), block.getZ());
        }

        @Nullable
        Block block() {
            World w = Bukkit.getWorld((String)this.world);
            if (w == null) {
                return null;
            }
            return w.getBlockAt(this.x, this.y, this.z);
        }
    }

    private record ParsedBook(List<PdcaStep> steps, Duration totalDuration, DisplayMode displayMode, AudienceMode audienceMode, Set<String> audienceNames) {
    }

    private static enum DisplayMode {
        BOTH,
        OVERALL_ONLY,
        PER_STEP_ONLY,
        NONE;

    }

    private static enum AudienceMode {
        NEARBY,
        ALL,
        LIST,
        NONE;

    }

    private record PdcaStep(int id, Duration duration, String label) {
    }
}

