/*
 * Decompiled with CFR 0.152.
 */
package com.clippyclip.clipsezrtp.rtp;

import com.clippyclip.clipsezrtp.config.ConfigManager;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.bukkit.Bukkit;
import org.bukkit.HeightMap;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.World;
import org.bukkit.WorldBorder;
import org.bukkit.block.Biome;
import org.bukkit.block.Block;
import org.bukkit.block.BlockFace;

public class SafeLocationFinder {
    private final ConfigManager configManager;
    private final Set<Material> safeGroundBlocks = new HashSet<Material>();
    private final Set<Biome> blacklistBiomes = new HashSet<Biome>();
    private static final int MAX_ATTEMPTS = 40;

    public SafeLocationFinder(ConfigManager configManager) {
        this.configManager = configManager;
        this.loadConfigData();
    }

    public void loadConfigData() {
        this.safeGroundBlocks.clear();
        this.blacklistBiomes.clear();
        List materialNames = this.configManager.getConfig().getStringList("safe-blocks");
        for (String name : materialNames) {
            try {
                Material mat = Material.valueOf((String)name.toUpperCase(Locale.ROOT));
                this.safeGroundBlocks.add(mat);
            }
            catch (IllegalArgumentException ex) {
                Bukkit.getLogger().warning("[ClipsEZRTP] Invalid material in safe-blocks: " + name);
            }
        }
        List biomeNames = this.configManager.getConfig().getStringList("blacklist-biomes");
        for (String name : biomeNames) {
            try {
                Biome biome = Biome.valueOf((String)name.toUpperCase(Locale.ROOT));
                this.blacklistBiomes.add(biome);
            }
            catch (IllegalArgumentException ex) {
                Bukkit.getLogger().warning("[ClipsEZRTP] Invalid biome in blacklist-biomes: " + name);
            }
        }
    }

    public void findSafeLocation(World world, Consumer<Location> onSuccess, Runnable onFail) {
        this.findSafeLocation(world, loc -> true, onSuccess, onFail);
    }

    public void findSafeLocation(World world, Predicate<Location> validator, Consumer<Location> onSuccess, Runnable onFail) {
        this.attempt(world, 0, validator == null ? loc -> true : validator, onSuccess, onFail);
    }

    private void attempt(World world, int attempt, Predicate<Location> validator, Consumer<Location> onSuccess, Runnable onFail) {
        if (attempt >= 40) {
            onFail.run();
            return;
        }
        Location randLoc = this.generateRandomLocation(world);
        if (randLoc == null) {
            onFail.run();
            return;
        }
        world.getChunkAtAsync(randLoc, true, chunk -> {
            Location safe = this.findSafeLocationInColumn(randLoc.getWorld(), randLoc.getBlockX(), randLoc.getBlockZ());
            if (safe != null && validator.test(safe)) {
                onSuccess.accept(safe);
            } else {
                this.attempt(world, attempt + 1, validator, onSuccess, onFail);
            }
        });
    }

    private Location generateRandomLocation(World world) {
        int minRadius = this.configManager.getMinRadius();
        int maxRadius = this.configManager.getMaxRadius();
        if (maxRadius < minRadius) {
            int tmp = minRadius;
            minRadius = maxRadius;
            maxRadius = tmp;
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        WorldBorder border = world.getWorldBorder();
        Location center = border != null ? border.getCenter() : world.getSpawnLocation();
        double borderRadius = border != null ? border.getSize() / 2.0 : (double)maxRadius;
        double effectiveMax = Math.min((double)maxRadius, borderRadius - 5.0);
        if (effectiveMax <= 0.0) {
            effectiveMax = maxRadius;
        }
        if (effectiveMax < (double)minRadius) {
            effectiveMax = minRadius;
        }
        double radius = (double)minRadius + (effectiveMax - (double)minRadius) * random.nextDouble();
        double angle = random.nextDouble() * 2.0 * Math.PI;
        double x = center.getX() + radius * Math.cos(angle);
        double z = center.getZ() + radius * Math.sin(angle);
        return new Location(world, x, (double)(world.getMaxHeight() - 1), z);
    }

    private Location findSafeLocationInColumn(World world, int x, int z) {
        boolean allowCaves = this.configManager.isAllowCaves();
        Biome biome = world.getBiome(x, world.getHighestBlockYAt(x, z), z);
        if (this.blacklistBiomes.contains(biome)) {
            return null;
        }
        if (!allowCaves) {
            Block highest = world.getHighestBlockAt(x, z, HeightMap.MOTION_BLOCKING_NO_LEAVES);
            int y = highest.getY();
            if (!this.isSafeGround(highest)) {
                return null;
            }
            Block feet = world.getBlockAt(x, y + 1, z);
            Block head = world.getBlockAt(x, y + 2, z);
            if (this.isSafeAir(feet) && this.isSafeAir(head)) {
                return new Location(world, (double)x + 0.5, (double)(y + 1), (double)z + 0.5);
            }
            return null;
        }
        int maxY = world.getMaxHeight() - 1;
        int minY = world.getMinHeight() + 1;
        for (int y = maxY; y >= minY; --y) {
            Block ground = world.getBlockAt(x, y, z);
            Block feet = ground.getRelative(BlockFace.UP);
            Block head = ground.getRelative(BlockFace.UP, 2);
            if (!this.isSafeGround(ground) || !this.isSafeAir(feet) || !this.isSafeAir(head)) continue;
            return new Location(world, (double)x + 0.5, (double)(y + 1), (double)z + 0.5);
        }
        return null;
    }

    private boolean isSafeGround(Block block) {
        Material type = block.getType();
        if (!this.safeGroundBlocks.contains(type)) {
            return false;
        }
        return type != Material.LAVA && type != Material.WATER && type != Material.CACTUS && type != Material.FIRE;
    }

    private boolean isSafeAir(Block block) {
        Material type = block.getType();
        String name = type.name();
        boolean isLiquidLike = name.contains("WATER") || name.contains("LAVA");
        return block.isPassable() && !isLiquidLike;
    }
}

