/*
 * Decompiled with CFR 0.152.
 */
package com.autobackup;

import com.autobackup.BackupCommand;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.command.CommandExecutor;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitTask;

public class AutoBackupPlugin
extends JavaPlugin {
    private BukkitTask backupTask;
    private Path backupDirectory;
    private Map<String, Long> fileTimestamps;
    private int backupIntervalMinutes;
    private int maxBackups;
    private boolean incrementalBackups;
    private boolean pauseDuringBackup;
    private boolean broadcastStart;
    private boolean broadcastComplete;
    private int compressionLevel;
    private List<String> excludePatterns;

    public void onEnable() {
        this.saveDefaultConfig();
        this.fileTimestamps = new HashMap<String, Long>();
        this.backupIntervalMinutes = this.getConfig().getInt("backup-interval-minutes", 60);
        this.maxBackups = this.getConfig().getInt("max-backups", 10);
        this.incrementalBackups = this.getConfig().getBoolean("incremental-backups", true);
        this.pauseDuringBackup = this.getConfig().getBoolean("pause-during-backup", false);
        this.broadcastStart = this.getConfig().getBoolean("broadcast-backup-start", false);
        this.broadcastComplete = this.getConfig().getBoolean("broadcast-backup-complete", false);
        this.compressionLevel = Math.max(0, Math.min(9, this.getConfig().getInt("compression-level", 6)));
        this.excludePatterns = this.getConfig().getStringList("exclude-patterns");
        File serverRoot = new File(System.getProperty("user.dir"));
        this.backupDirectory = serverRoot.toPath().resolve("backups");
        try {
            Files.createDirectories(this.backupDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            this.getLogger().severe("Failed to create backup directory: " + e.getMessage());
            return;
        }
        this.getCommand("autobackup").setExecutor((CommandExecutor)new BackupCommand(this));
        this.scheduleBackups();
        this.getLogger().info("AutoBackup enabled! Backups every " + this.backupIntervalMinutes + " minutes.");
        this.getLogger().info("Backup location: " + this.backupDirectory.toString());
    }

    public void onDisable() {
        if (this.backupTask != null) {
            this.backupTask.cancel();
        }
        this.getLogger().info("AutoBackup disabled!");
    }

    private void scheduleBackups() {
        long intervalTicks = (long)(this.backupIntervalMinutes * 60) * 20L;
        this.backupTask = Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this, () -> this.performBackup(false), intervalTicks, intervalTicks);
    }

    public void performBackup(boolean manual) {
        String timestamp = new SimpleDateFormat("yyyy-MM-dd_HH-mm-ss").format(new Date());
        String backupType = this.incrementalBackups && !manual ? "incremental" : "full";
        String backupName = "backup_" + timestamp + "_" + backupType + ".zip";
        Path backupFile = this.backupDirectory.resolve(backupName);
        this.getLogger().info("Starting " + backupType + " backup...");
        if (this.broadcastStart) {
            Bukkit.broadcastMessage((String)"\u00a7e[AutoBackup] Creating backup...");
        }
        Bukkit.getScheduler().runTask((Plugin)this, () -> {
            for (World world : Bukkit.getWorlds()) {
                world.save();
            }
            if (this.pauseDuringBackup) {
                for (World world : Bukkit.getWorlds()) {
                    world.setAutoSave(false);
                }
                this.getLogger().info("Auto-save paused during backup.");
            }
            this.getLogger().info("All worlds saved and ready for backup.");
            CompletableFuture.runAsync(() -> {
                block5: {
                    try {
                        this.createBackupZip(backupFile, manual);
                        this.cleanOldBackups();
                        this.getLogger().info("Backup completed: " + backupName);
                        if (this.pauseDuringBackup) {
                            Bukkit.getScheduler().runTask((Plugin)this, () -> {
                                for (World world : Bukkit.getWorlds()) {
                                    world.setAutoSave(true);
                                }
                                this.getLogger().info("Auto-save re-enabled.");
                            });
                        }
                        if (this.broadcastComplete) {
                            Bukkit.getScheduler().runTask((Plugin)this, () -> Bukkit.broadcastMessage((String)"\u00a7a[AutoBackup] Backup completed!"));
                        }
                        if (manual) {
                            Bukkit.getScheduler().runTask((Plugin)this, () -> Bukkit.broadcastMessage((String)"\u00a7a[AutoBackup] Manual backup completed successfully!"));
                        }
                    }
                    catch (IOException e) {
                        this.getLogger().severe("Backup failed: " + e.getMessage());
                        e.printStackTrace();
                        if (!this.pauseDuringBackup) break block5;
                        Bukkit.getScheduler().runTask((Plugin)this, () -> {
                            for (World world : Bukkit.getWorlds()) {
                                world.setAutoSave(true);
                            }
                        });
                    }
                }
            });
        });
    }

    private void createBackupZip(Path zipFile, boolean forceFull) throws IOException {
        Path serverRoot = new File(System.getProperty("user.dir")).toPath();
        HashSet<String> changedFiles = new HashSet<String>();
        try (ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(zipFile.toFile()));){
            zos.setLevel(this.compressionLevel);
            List worldNames = this.getConfig().getStringList("worlds-to-backup");
            if (worldNames.isEmpty()) {
                for (World world : Bukkit.getWorlds()) {
                    worldNames.add(world.getName());
                }
            }
            for (String worldName : worldNames) {
                Path worldPath = serverRoot.resolve(worldName);
                if (!Files.exists(worldPath, new LinkOption[0])) continue;
                this.addDirectoryToZip(zos, serverRoot, worldPath, changedFiles, forceFull);
            }
            Path pluginsPath = serverRoot.resolve("plugins");
            this.addDirectoryToZip(zos, serverRoot, pluginsPath, changedFiles, forceFull);
            List configPaths = this.getConfig().getStringList("additional-paths");
            configPaths.addAll(Arrays.asList("server.properties", "bukkit.yml", "spigot.yml", "paper.yml", "paper-global.yml", "paper-world-defaults.yml", "config"));
            for (String configPath : configPaths) {
                Path path = serverRoot.resolve(configPath);
                if (!Files.exists(path, new LinkOption[0])) continue;
                if (Files.isDirectory(path, new LinkOption[0])) {
                    this.addDirectoryToZip(zos, serverRoot, path, changedFiles, forceFull);
                    continue;
                }
                this.addFileToZip(zos, serverRoot, path, changedFiles, forceFull);
            }
            if (this.incrementalBackups && !forceFull) {
                this.getLogger().info("Incremental backup: " + changedFiles.size() + " files changed");
            }
        }
    }

    private void addDirectoryToZip(final ZipOutputStream zos, final Path rootPath, Path dirPath, final Set<String> changedFiles, final boolean forceFull) throws IOException {
        Files.walkFileTree(dirPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String fileName = file.getFileName().toString();
                if (fileName.equals("session.lock") || fileName.endsWith(".tmp") || fileName.endsWith(".temp")) {
                    return FileVisitResult.CONTINUE;
                }
                String relativePath = rootPath.relativize(file).toString().replace('\\', '/');
                if (AutoBackupPlugin.this.shouldExclude(relativePath)) {
                    return FileVisitResult.CONTINUE;
                }
                AutoBackupPlugin.this.addFileToZip(zos, rootPath, file, changedFiles, forceFull);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                String dirName = dir.getFileName().toString();
                if (dirName.equals("cache") || dirName.equals("logs") || dirName.equals("crash-reports")) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                String relativePath = rootPath.relativize(dir).toString().replace('\\', '/');
                if (AutoBackupPlugin.this.shouldExclude(relativePath)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void addFileToZip(ZipOutputStream zos, Path rootPath, Path file, Set<String> changedFiles, boolean forceFull) throws IOException {
        Long previousTimestamp;
        String relativePath = rootPath.relativize(file).toString().replace('\\', '/');
        long lastModified = Files.getLastModifiedTime(file, new LinkOption[0]).toMillis();
        if (!forceFull && this.incrementalBackups && (previousTimestamp = this.fileTimestamps.get(relativePath)) != null && previousTimestamp >= lastModified) {
            return;
        }
        ZipEntry entry = new ZipEntry(relativePath);
        entry.setTime(lastModified);
        zos.putNextEntry(entry);
        Files.copy(file, zos);
        zos.closeEntry();
        this.fileTimestamps.put(relativePath, lastModified);
        changedFiles.add(relativePath);
    }

    private boolean shouldExclude(String relativePath) {
        for (String pattern : this.excludePatterns) {
            if (!this.matchesGlob(relativePath, pattern)) continue;
            return true;
        }
        return false;
    }

    private boolean matchesGlob(String path, String pattern) {
        String regex = pattern.replace("\\", "/").replace(".", "\\.").replace("**", "DOUBLE_STAR").replace("*", "[^/]*").replace("DOUBLE_STAR", ".*");
        return path.matches(regex);
    }

    private void cleanOldBackups() {
        try {
            ArrayList<Path> backups = new ArrayList<Path>();
            Files.list(this.backupDirectory).filter(p -> p.getFileName().toString().startsWith("backup_")).filter(p -> p.getFileName().toString().endsWith(".zip")).forEach(backups::add);
            if (backups.size() > this.maxBackups) {
                backups.sort(Comparator.comparingLong(p -> {
                    try {
                        return Files.getLastModifiedTime(p, new LinkOption[0]).toMillis();
                    }
                    catch (IOException e) {
                        return 0L;
                    }
                }));
                int toDelete = backups.size() - this.maxBackups;
                for (int i = 0; i < toDelete; ++i) {
                    Files.delete((Path)backups.get(i));
                    this.getLogger().info("Deleted old backup: " + String.valueOf(((Path)backups.get(i)).getFileName()));
                }
            }
        }
        catch (IOException e) {
            this.getLogger().warning("Failed to clean old backups: " + e.getMessage());
        }
    }

    public void reloadPluginConfig() {
        this.reloadConfig();
        this.backupIntervalMinutes = this.getConfig().getInt("backup-interval-minutes", 60);
        this.maxBackups = this.getConfig().getInt("max-backups", 10);
        this.incrementalBackups = this.getConfig().getBoolean("incremental-backups", true);
        this.pauseDuringBackup = this.getConfig().getBoolean("pause-during-backup", false);
        this.broadcastStart = this.getConfig().getBoolean("broadcast-backup-start", false);
        this.broadcastComplete = this.getConfig().getBoolean("broadcast-backup-complete", false);
        this.compressionLevel = Math.max(0, Math.min(9, this.getConfig().getInt("compression-level", 6)));
        this.excludePatterns = this.getConfig().getStringList("exclude-patterns");
        if (this.backupTask != null) {
            this.backupTask.cancel();
        }
        this.scheduleBackups();
    }

    public Path getBackupDirectory() {
        return this.backupDirectory;
    }
}

