/*
 * Decompiled with CFR 0.152.
 */
package us.ironcladnetwork.blockback;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;

public class PlayerDataManager {
    private static PlayerDataManager instance;
    private final JavaPlugin plugin;
    private final File configFile;
    private FileConfiguration config;
    private final AtomicBoolean saveInProgress = new AtomicBoolean(false);
    private final AtomicBoolean pendingSave = new AtomicBoolean(false);
    private volatile CountDownLatch shutdownLatch;
    private static final int MAX_CACHE_SIZE = 100;
    private static final long CACHE_EXPIRY_MINUTES = 30L;
    private static final long CACHE_CLEANUP_INTERVAL_TICKS = 6000L;
    private final ConcurrentHashMap<UUID, PlayerSettings> playerCache = new ConcurrentHashMap();
    private int cacheCleanupTaskId = -1;

    public static void init(JavaPlugin plugin) {
        instance = new PlayerDataManager(plugin);
    }

    public static PlayerDataManager getInstance() {
        return instance;
    }

    private PlayerDataManager(JavaPlugin plugin) {
        this.plugin = plugin;
        File dataFolder = plugin.getDataFolder();
        if (!dataFolder.exists()) {
            dataFolder.mkdirs();
        }
        this.configFile = new File(dataFolder, "players.yml");
        if (!this.configFile.exists()) {
            try {
                this.configFile.createNewFile();
            }
            catch (IOException e) {
                plugin.getLogger().severe("Could not create players.yml: " + e.getMessage());
            }
        }
        this.loadConfiguration();
        this.cleanupOrphanedTempFiles();
        this.startCacheCleanupTask();
    }

    private void setDefaults(String uuid, String name) {
        this.config.set(uuid + ".name", (Object)name);
        this.config.set(uuid + ".barkback", (Object)true);
        this.config.set(uuid + ".pathback", (Object)true);
        this.config.set(uuid + ".farmback", (Object)true);
    }

    private void loadConfiguration() {
        block4: {
            try {
                this.config = YamlConfiguration.loadConfiguration((File)this.configFile);
                if (!this.validateConfiguration()) {
                    this.plugin.getLogger().warning("Configuration validation failed, attempting recovery...");
                    if (!this.recoverFromBackup()) {
                        this.plugin.getLogger().warning("Recovery failed, creating new configuration with defaults");
                        this.createEmptyConfiguration();
                    }
                }
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Failed to load players.yml: " + e.getMessage());
                this.plugin.getLogger().warning("Attempting to recover from backup...");
                if (this.recoverFromBackup()) break block4;
                this.plugin.getLogger().warning("Recovery failed, creating new configuration");
                this.createEmptyConfiguration();
            }
        }
    }

    private boolean validateConfiguration() {
        try {
            if (this.config == null) {
                return false;
            }
            this.config.getKeys(false);
            for (String key : this.config.getKeys(false)) {
                if (key == null || key.isEmpty()) continue;
                this.config.getConfigurationSection(key);
            }
            return true;
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Configuration validation failed: " + e.getMessage());
            return false;
        }
    }

    private void createBackup() {
        if (!this.configFile.exists()) {
            return;
        }
        try {
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss"));
            File backupFile = new File(this.configFile.getParent(), "players.yml.backup." + timestamp);
            Files.copy(this.configFile.toPath(), backupFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            this.plugin.getLogger().info("Created backup: " + backupFile.getName());
            this.cleanupOldBackups();
        }
        catch (IOException e) {
            this.plugin.getLogger().warning("Failed to create backup: " + e.getMessage());
        }
    }

    private void cleanupOldBackups() {
        File dataFolder = this.configFile.getParentFile();
        File[] backups = dataFolder.listFiles((dir, name) -> name.startsWith("players.yml.backup."));
        if (backups != null && backups.length > 5) {
            Arrays.sort(backups, (a, b) -> Long.compare(a.lastModified(), b.lastModified()));
            for (int i = 0; i < backups.length - 5; ++i) {
                if (!backups[i].delete()) continue;
                this.plugin.getLogger().info("Deleted old backup: " + backups[i].getName());
            }
        }
    }

    private boolean recoverFromBackup() {
        File dataFolder = this.configFile.getParentFile();
        File[] backups = dataFolder.listFiles((dir, name) -> name.startsWith("players.yml.backup."));
        if (backups == null || backups.length == 0) {
            this.plugin.getLogger().warning("No backup files found for recovery");
            return false;
        }
        Arrays.sort(backups, (a, b) -> Long.compare(b.lastModified(), a.lastModified()));
        for (File backup : backups) {
            try {
                this.plugin.getLogger().info("Attempting recovery from: " + backup.getName());
                YamlConfiguration backupConfig = YamlConfiguration.loadConfiguration((File)backup);
                backupConfig.getKeys(false);
                Files.copy(backup.toPath(), this.configFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                this.config = YamlConfiguration.loadConfiguration((File)this.configFile);
                this.plugin.getLogger().info("Successfully recovered from backup: " + backup.getName());
                return true;
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Failed to recover from " + backup.getName() + ": " + e.getMessage());
            }
        }
        this.plugin.getLogger().severe("All backup recovery attempts failed");
        return false;
    }

    private void createEmptyConfiguration() {
        this.config = new YamlConfiguration();
        this.plugin.getLogger().info("Created new empty configuration file");
        this.saveConfig();
    }

    private PlayerSettings loadAndValidatePlayerSettings(String uuid, String playerName) {
        try {
            String name = this.config.getString(uuid + ".name");
            Object barkbackObj = this.config.get(uuid + ".barkback");
            Object pathbackObj = this.config.get(uuid + ".pathback");
            Object farmbackObj = this.config.get(uuid + ".farmback");
            boolean barkback = this.validateBooleanSetting(barkbackObj, true, "barkback", uuid);
            boolean pathback = this.validateBooleanSetting(pathbackObj, true, "pathback", uuid);
            boolean farmback = this.validateBooleanSetting(farmbackObj, true, "farmback", uuid);
            if (name == null || name.trim().isEmpty()) {
                name = playerName;
                this.config.set(uuid + ".name", (Object)name);
                this.saveConfig();
            }
            return new PlayerSettings(name, barkback, pathback, farmback);
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to load settings for player " + uuid + ": " + e.getMessage());
            this.plugin.getLogger().info("Using default settings for player " + playerName);
            PlayerSettings defaults = new PlayerSettings(playerName);
            this.setDefaults(uuid, playerName);
            this.saveConfig();
            return defaults;
        }
    }

    private boolean validateBooleanSetting(Object value, boolean defaultValue, String settingName, String uuid) {
        if (value == null) {
            this.plugin.getLogger().warning("Missing " + settingName + " setting for player " + uuid + ", using default: " + defaultValue);
            return defaultValue;
        }
        if (value instanceof Boolean) {
            return (Boolean)value;
        }
        if (value instanceof String) {
            String strValue = ((String)value).trim();
            if (strValue.isEmpty()) {
                this.plugin.getLogger().warning("Empty " + settingName + " setting for player " + uuid + ", using default: " + defaultValue);
                return defaultValue;
            }
            if ("true".equals(strValue = strValue.replaceAll("[^a-zA-Z0-9]", "").toLowerCase()) || "yes".equals(strValue) || "y".equals(strValue) || "1".equals(strValue) || "on".equals(strValue) || "enabled".equals(strValue) || "enable".equals(strValue) || "t".equals(strValue)) {
                return true;
            }
            if ("false".equals(strValue) || "no".equals(strValue) || "n".equals(strValue) || "0".equals(strValue) || "off".equals(strValue) || "disabled".equals(strValue) || "disable".equals(strValue) || "f".equals(strValue)) {
                return false;
            }
            this.plugin.getLogger().warning("Invalid " + settingName + " value '" + String.valueOf(value) + "' for player " + uuid + ", using default: " + defaultValue);
            return defaultValue;
        }
        if (value instanceof Number) {
            return ((Number)value).intValue() != 0;
        }
        try {
            String strValue = value.toString().trim();
            return this.validateBooleanSetting(strValue, defaultValue, settingName, uuid);
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Cannot parse " + settingName + " value '" + String.valueOf(value) + "' (type: " + value.getClass().getSimpleName() + ") for player " + uuid + ", using default: " + defaultValue);
            return defaultValue;
        }
    }

    private PlayerSettings getEmergencyDefaults(Player player) {
        this.plugin.getLogger().warning("Using emergency defaults for player " + player.getName() + " due to data corruption");
        return new PlayerSettings(player.getName(), true, true, true);
    }

    public boolean isConfigurationHealthy() {
        try {
            return this.config != null && this.configFile != null && this.configFile.getParentFile().canWrite();
        }
        catch (Exception e) {
            return false;
        }
    }

    private boolean getFeatureSetting(Player player, String featureName) {
        PlayerSettings settings;
        UUID uuid = player.getUniqueId();
        PlayerSettings cached = this.playerCache.get(uuid);
        if (cached != null) {
            cached.updateLastAccessed();
            switch (featureName) {
                case "barkback": {
                    return cached.barkback;
                }
                case "pathback": {
                    return cached.pathback;
                }
                case "farmback": {
                    return cached.farmback;
                }
            }
            return true;
        }
        String uuidStr = uuid.toString();
        try {
            if (!this.config.contains(uuidStr)) {
                settings = new PlayerSettings(player.getName());
                this.setDefaults(uuidStr, player.getName());
                this.saveConfig();
            } else {
                settings = this.loadAndValidatePlayerSettings(uuidStr, player.getName());
            }
            if (this.playerCache.size() >= 100) {
                this.playerCache.entrySet().stream().min((e1, e2) -> Long.compare(((PlayerSettings)e1.getValue()).lastAccessed, ((PlayerSettings)e2.getValue()).lastAccessed)).ifPresent(entry -> this.playerCache.remove(entry.getKey()));
            }
            this.playerCache.put(uuid, settings);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Critical error loading player settings for " + player.getName() + ": " + e.getMessage());
            settings = this.getEmergencyDefaults(player);
            this.playerCache.put(uuid, settings);
        }
        switch (featureName) {
            case "barkback": {
                return settings.barkback;
            }
            case "pathback": {
                return settings.pathback;
            }
            case "farmback": {
                return settings.farmback;
            }
        }
        return true;
    }

    private void setFeatureSetting(Player player, String featureName, boolean enabled) {
        UUID uuid = player.getUniqueId();
        String uuidStr = uuid.toString();
        this.config.set(uuidStr + ".name", (Object)player.getName());
        this.config.set(uuidStr + "." + featureName, (Object)enabled);
        this.saveConfig();
        PlayerSettings cached = this.playerCache.get(uuid);
        if (cached == null) {
            cached = new PlayerSettings(player.getName());
            if (this.playerCache.size() >= 100) {
                this.playerCache.entrySet().stream().min((e1, e2) -> Long.compare(((PlayerSettings)e1.getValue()).lastAccessed, ((PlayerSettings)e2.getValue()).lastAccessed)).ifPresent(entry -> this.playerCache.remove(entry.getKey()));
            }
            this.playerCache.put(uuid, cached);
        } else {
            cached.updateLastAccessed();
        }
        switch (featureName) {
            case "barkback": {
                cached.barkback = enabled;
                break;
            }
            case "pathback": {
                cached.pathback = enabled;
                break;
            }
            case "farmback": {
                cached.farmback = enabled;
            }
        }
        cached.name = player.getName();
    }

    public boolean isBarkBackEnabled(Player player) {
        return this.getFeatureSetting(player, "barkback");
    }

    public boolean isPathBackEnabled(Player player) {
        return this.getFeatureSetting(player, "pathback");
    }

    public boolean isFarmBackEnabled(Player player) {
        return this.getFeatureSetting(player, "farmback");
    }

    public void setBarkBack(Player player, boolean enabled) {
        this.setFeatureSetting(player, "barkback", enabled);
    }

    public void setPathBack(Player player, boolean enabled) {
        this.setFeatureSetting(player, "pathback", enabled);
    }

    public void setFarmBack(Player player, boolean enabled) {
        this.setFeatureSetting(player, "farmback", enabled);
    }

    private void saveConfig() {
        this.pendingSave.set(true);
        if (!this.saveInProgress.compareAndSet(false, true)) {
            return;
        }
        this.plugin.getServer().getScheduler().runTaskAsynchronously((Plugin)this.plugin, () -> {
            File tempFile = null;
            try {
                do {
                    this.pendingSave.set(false);
                    tempFile = new File(this.configFile.getAbsolutePath() + ".tmp");
                    boolean tempFileCreated = false;
                    try {
                        if (this.configFile.exists() && this.configFile.length() > 0L) {
                            this.createBackup();
                        }
                        tempFileCreated = true;
                        this.config.save(tempFile);
                        if (!tempFile.renameTo(this.configFile)) {
                            this.plugin.getLogger().warning("Atomic rename failed, falling back to direct save");
                            this.config.save(this.configFile);
                            if (tempFile.exists() && !tempFile.delete()) {
                                tempFile.deleteOnExit();
                                this.plugin.getLogger().warning("Failed to delete temporary file, scheduled for deletion on exit: " + tempFile.getAbsolutePath());
                            }
                        }
                        tempFile = null;
                    }
                    catch (IOException saveException) {
                        this.plugin.getLogger().severe("Failed to save to temporary file: " + saveException.getMessage());
                        if (tempFileCreated && tempFile != null && tempFile.exists() && !tempFile.delete()) {
                            tempFile.deleteOnExit();
                        }
                        try {
                            this.config.save(this.configFile);
                        }
                        catch (IOException directSaveException) {
                            this.plugin.getLogger().severe("Direct save also failed: " + directSaveException.getMessage());
                            throw directSaveException;
                        }
                    }
                    finally {
                        if (tempFile != null && tempFile.exists() && !tempFile.delete()) {
                            tempFile.deleteOnExit();
                            this.plugin.getLogger().warning("Failed to delete temporary file, scheduled for deletion on exit: " + tempFile.getAbsolutePath());
                        }
                    }
                } while (this.pendingSave.compareAndSet(true, false));
            }
            catch (IOException e) {
                this.plugin.getLogger().severe("Could not save players.yml asynchronously: " + e.getMessage());
            }
            finally {
                if (tempFile != null && tempFile.exists() && !tempFile.delete()) {
                    tempFile.deleteOnExit();
                }
                this.saveInProgress.set(false);
                CountDownLatch latch = this.shutdownLatch;
                if (latch != null) {
                    latch.countDown();
                }
                if (this.pendingSave.get()) {
                    this.saveConfig();
                }
            }
        });
    }

    public void reloadConfig() {
        this.playerCache.clear();
        try {
            this.loadConfiguration();
            this.plugin.getLogger().info("Player configuration reloaded successfully");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to reload player configuration: " + e.getMessage());
            this.plugin.getLogger().warning("Player data may be using fallback defaults until next restart");
        }
    }

    public void removeFromCache(Player player) {
        this.playerCache.remove(player.getUniqueId());
    }

    public void removeFromCache(UUID uuid) {
        this.playerCache.remove(uuid);
    }

    public int getCacheSize() {
        return this.playerCache.size();
    }

    public void clearCache() {
        this.playerCache.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean shutdown(int timeoutSeconds) {
        this.stopCacheCleanupTask();
        if (!this.saveInProgress.get() && !this.pendingSave.get()) {
            return true;
        }
        this.shutdownLatch = new CountDownLatch(1);
        try {
            boolean completed = this.shutdownLatch.await(timeoutSeconds, TimeUnit.SECONDS);
            if (!completed) {
                this.plugin.getLogger().warning("Timeout waiting for player data save to complete during shutdown");
            }
            boolean bl = completed;
            return bl;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.plugin.getLogger().warning("Interrupted while waiting for player data save during shutdown");
            boolean bl = false;
            return bl;
        }
        finally {
            this.shutdownLatch = null;
        }
    }

    private void cleanupOrphanedTempFiles() {
        File[] tempFiles;
        File dataFolder = this.configFile.getParentFile();
        if (dataFolder.exists() && dataFolder.isDirectory() && (tempFiles = dataFolder.listFiles((dir, name) -> name.endsWith(".tmp"))) != null && tempFiles.length > 0) {
            for (File tempFile : tempFiles) {
                if (tempFile.delete()) {
                    this.plugin.getLogger().info("Cleaned up orphaned temp file: " + tempFile.getName());
                    continue;
                }
                tempFile.deleteOnExit();
                this.plugin.getLogger().warning("Failed to delete orphaned temp file, scheduled for deletion on exit: " + tempFile.getName());
            }
        }
    }

    private void startCacheCleanupTask() {
        this.cacheCleanupTaskId = this.plugin.getServer().getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> this.cleanupCache(), 6000L, 6000L).getTaskId();
    }

    public void stopCacheCleanupTask() {
        if (this.cacheCleanupTaskId != -1) {
            this.plugin.getServer().getScheduler().cancelTask(this.cacheCleanupTaskId);
            this.cacheCleanupTaskId = -1;
        }
    }

    private void cleanupCache() {
        long now = System.currentTimeMillis();
        long expiryTime = 1800000L;
        Iterator<Map.Entry<UUID, PlayerSettings>> iterator = this.playerCache.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<UUID, PlayerSettings> entry2 = iterator.next();
            PlayerSettings settings = entry2.getValue();
            if (this.plugin.getServer().getPlayer(entry2.getKey()) != null) {
                settings.updateLastAccessed();
                continue;
            }
            if (now - settings.lastAccessed <= expiryTime) continue;
            iterator.remove();
            this.plugin.getLogger().fine("Removed expired cache entry for player: " + settings.name);
        }
        if (this.playerCache.size() > 100) {
            this.playerCache.entrySet().stream().sorted((e1, e2) -> Long.compare(((PlayerSettings)e1.getValue()).lastAccessed, ((PlayerSettings)e2.getValue()).lastAccessed)).limit(this.playerCache.size() - 100).forEach(entry -> {
                this.playerCache.remove(entry.getKey());
                this.plugin.getLogger().fine("Removed cache entry due to size limit: " + ((PlayerSettings)entry.getValue()).name);
            });
        }
    }

    public static class PlayerSettings {
        public boolean barkback = true;
        public boolean pathback = true;
        public boolean farmback = true;
        public String name;
        public long lastAccessed;

        public PlayerSettings(String name) {
            this.name = name;
            this.lastAccessed = System.currentTimeMillis();
        }

        public PlayerSettings(String name, boolean barkback, boolean pathback, boolean farmback) {
            this.name = name;
            this.barkback = barkback;
            this.pathback = pathback;
            this.farmback = farmback;
            this.lastAccessed = System.currentTimeMillis();
        }

        public void updateLastAccessed() {
            this.lastAccessed = System.currentTimeMillis();
        }
    }
}

