/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.ezshops.shop;

import com.skyblockexp.ezshops.gui.ShopMenu;
import com.skyblockexp.ezshops.shop.ShopPricingManager;
import com.skyblockexp.ezshops.shop.ShopRotationDefinition;
import com.skyblockexp.ezshops.shop.ShopRotationMode;
import com.skyblockexp.ezshops.shop.ShopRotationOption;
import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import java.util.logging.Level;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitTask;

public final class ShopRotationManager {
    private final JavaPlugin plugin;
    private final ShopPricingManager pricingManager;
    private final ShopMenu shopMenu;
    private final File stateFile;
    private final Map<String, RotationState> rotationStates = new LinkedHashMap<String, RotationState>();
    private BukkitTask task;

    public ShopRotationManager(JavaPlugin plugin, ShopPricingManager pricingManager, ShopMenu shopMenu) {
        this.plugin = Objects.requireNonNull(plugin, "plugin");
        this.pricingManager = Objects.requireNonNull(pricingManager, "pricingManager");
        this.shopMenu = shopMenu;
        this.stateFile = new File(plugin.getDataFolder(), "shop-rotations.yml");
    }

    public void enable() {
        this.loadState();
        Map<String, ShopRotationDefinition> definitions = this.pricingManager.getRotationDefinitions();
        Map<String, String> activeOptions = this.pricingManager.getActiveRotationOptions();
        long now = System.currentTimeMillis();
        UpdateResult syncResult = this.synchronizeStates(definitions, activeOptions, now);
        if (syncResult.stateChanged()) {
            this.saveState();
        }
        if (this.shopMenu != null && syncResult.layoutChanged()) {
            this.shopMenu.refreshViewers();
        }
        if (this.task != null) {
            this.task.cancel();
        }
        this.task = this.plugin.getServer().getScheduler().runTaskTimer((Plugin)this.plugin, this::tick, 20L, 20L);
    }

    public void disable() {
        if (this.task != null) {
            this.task.cancel();
            this.task = null;
        }
        this.saveState();
        this.rotationStates.clear();
    }

    private void tick() {
        boolean layoutChanged;
        Map<String, ShopRotationDefinition> definitions = this.pricingManager.getRotationDefinitions();
        Map<String, String> activeOptions = this.pricingManager.getActiveRotationOptions();
        long now = System.currentTimeMillis();
        UpdateResult syncResult = this.synchronizeStates(definitions, activeOptions, now);
        UpdateResult advanceResult = this.processDueRotations(definitions, now);
        boolean stateChanged = syncResult.stateChanged() || advanceResult.stateChanged();
        boolean bl = layoutChanged = syncResult.layoutChanged() || advanceResult.layoutChanged();
        if (stateChanged) {
            this.saveState();
        }
        if (layoutChanged && this.shopMenu != null) {
            this.shopMenu.refreshViewers();
        }
    }

    private UpdateResult synchronizeStates(Map<String, ShopRotationDefinition> definitions, Map<String, String> activeOptions, long now) {
        boolean stateChanged = false;
        boolean layoutChanged = false;
        if (this.rotationStates.keySet().removeIf(id -> !definitions.containsKey(id))) {
            stateChanged = true;
        }
        for (Map.Entry<String, ShopRotationDefinition> entry : definitions.entrySet()) {
            Long targetNext;
            String resolvedActive;
            String savedActive;
            String rotationId = entry.getKey();
            ShopRotationDefinition definition = entry.getValue();
            RotationState state = this.rotationStates.computeIfAbsent(rotationId, key -> new RotationState());
            String previousActive = state.activeOptionId;
            Long previousNext = state.nextChangeMillis;
            String pricingActive = activeOptions.get(rotationId);
            if (pricingActive != null && !definition.containsOption(pricingActive)) {
                pricingActive = null;
            }
            if ((savedActive = state.activeOptionId) != null && !definition.containsOption(savedActive)) {
                savedActive = null;
            }
            String string = pricingActive != null ? pricingActive : (resolvedActive = savedActive != null ? savedActive : definition.defaultOptionId());
            if (!Objects.equals(pricingActive, resolvedActive)) {
                if (!this.pricingManager.setActiveRotationOption(rotationId, resolvedActive)) {
                    resolvedActive = definition.defaultOptionId();
                    this.pricingManager.setActiveRotationOption(rotationId, resolvedActive);
                }
                pricingActive = resolvedActive;
                layoutChanged = true;
            }
            if (!Objects.equals(previousActive, pricingActive)) {
                state.activeOptionId = pricingActive;
                stateChanged = true;
                layoutChanged = true;
            }
            if (Objects.equals(previousNext, targetNext = !Objects.equals(previousActive, pricingActive) ? this.computeNextChange(definition, now) : this.normalizeNextChange(definition, previousNext, now))) continue;
            state.nextChangeMillis = targetNext;
            stateChanged = true;
        }
        return new UpdateResult(stateChanged, layoutChanged);
    }

    private UpdateResult processDueRotations(Map<String, ShopRotationDefinition> definitions, long now) {
        boolean stateChanged = false;
        boolean layoutChanged = false;
        for (Map.Entry<String, ShopRotationDefinition> entry : definitions.entrySet()) {
            Long newNext;
            String nextOption;
            Long nextChange;
            String rotationId = entry.getKey();
            ShopRotationDefinition definition = entry.getValue();
            RotationState state = this.rotationStates.get(rotationId);
            if (state == null || (nextChange = state.nextChangeMillis) == null || nextChange > now) continue;
            String currentOption = state.activeOptionId;
            if (currentOption == null || !definition.containsOption(currentOption)) {
                currentOption = definition.defaultOptionId();
            }
            if (!definition.containsOption(nextOption = this.determineNextOption(definition, currentOption))) {
                nextOption = definition.defaultOptionId();
            }
            if (!Objects.equals(currentOption, nextOption)) {
                if (!this.pricingManager.setActiveRotationOption(rotationId, nextOption)) {
                    nextOption = definition.defaultOptionId();
                    this.pricingManager.setActiveRotationOption(rotationId, nextOption);
                }
                state.activeOptionId = nextOption;
                layoutChanged = true;
                stateChanged = true;
                this.plugin.getLogger().info("Rotation '" + rotationId + "' advanced to option '" + nextOption + "'.");
            }
            if (Objects.equals(state.nextChangeMillis, newNext = this.computeNextChange(definition, now))) continue;
            state.nextChangeMillis = newNext;
            stateChanged = true;
        }
        return new UpdateResult(stateChanged, layoutChanged);
    }

    private String determineNextOption(ShopRotationDefinition definition, String currentOption) {
        return switch (definition.mode()) {
            case ShopRotationMode.SEQUENTIAL -> this.pickSequential(definition, currentOption);
            case ShopRotationMode.RANDOM -> this.pickRandom(definition, currentOption);
            default -> throw new IncompatibleClassChangeError();
        };
    }

    private String pickSequential(ShopRotationDefinition definition, String currentOption) {
        List<ShopRotationOption> options = definition.options();
        if (options.isEmpty()) {
            return currentOption;
        }
        int currentIndex = -1;
        int i = 0;
        while (i < options.size()) {
            if (options.get(i).id().equalsIgnoreCase(currentOption)) {
                currentIndex = i;
                break;
            }
            ++i;
        }
        int nextIndex = currentIndex >= 0 ? (currentIndex + 1) % options.size() : 0;
        return options.get(nextIndex).id();
    }

    private String pickRandom(ShopRotationDefinition definition, String currentOption) {
        List<ShopRotationOption> options = definition.options();
        if (options.isEmpty()) {
            return currentOption;
        }
        double totalWeight = 0.0;
        for (ShopRotationOption option : options) {
            totalWeight += Math.max(0.0, option.weight());
        }
        if (totalWeight <= 0.0) {
            return options.get(ThreadLocalRandom.current().nextInt(options.size())).id();
        }
        double selection = ThreadLocalRandom.current().nextDouble(totalWeight);
        double cumulative = 0.0;
        for (ShopRotationOption option : options) {
            double weight = Math.max(0.0, option.weight());
            if (weight <= 0.0 || !(selection < (cumulative += weight))) continue;
            return option.id();
        }
        return options.get(options.size() - 1).id();
    }

    private Long computeNextChange(ShopRotationDefinition definition, long now) {
        Duration interval = definition.interval();
        if (interval == null || interval.isZero() || interval.isNegative()) {
            return null;
        }
        long delay = Math.max(1L, interval.toMillis());
        return now + delay;
    }

    private Long normalizeNextChange(ShopRotationDefinition definition, Long currentValue, long now) {
        Duration interval = definition.interval();
        if (interval == null || interval.isZero() || interval.isNegative()) {
            return null;
        }
        if (currentValue != null && currentValue > now) {
            return currentValue;
        }
        return this.computeNextChange(definition, now);
    }

    private void loadState() {
        this.rotationStates.clear();
        if (!this.stateFile.exists()) {
            return;
        }
        YamlConfiguration configuration = YamlConfiguration.loadConfiguration((File)this.stateFile);
        ConfigurationSection rotationsSection = configuration.getConfigurationSection("rotations");
        if (rotationsSection == null) {
            return;
        }
        for (String rotationId : rotationsSection.getKeys(false)) {
            long stored;
            ConfigurationSection section = rotationsSection.getConfigurationSection(rotationId);
            if (section == null) continue;
            String activeOption = section.getString("active-option");
            Long nextChange = null;
            if (section.contains("next-change") && (stored = section.getLong("next-change")) > 0L) {
                nextChange = stored;
            }
            this.rotationStates.put(rotationId, new RotationState(activeOption, nextChange));
        }
    }

    private void saveState() {
        File dataFolder = this.plugin.getDataFolder();
        if (!dataFolder.exists() && !dataFolder.mkdirs()) {
            this.plugin.getLogger().warning("Unable to create plugin data folder for shop rotation state.");
            return;
        }
        YamlConfiguration configuration = new YamlConfiguration();
        if (!this.rotationStates.isEmpty()) {
            ConfigurationSection rotationsSection = configuration.createSection("rotations");
            for (Map.Entry<String, RotationState> entry : this.rotationStates.entrySet()) {
                RotationState state = entry.getValue();
                ConfigurationSection section = rotationsSection.createSection(entry.getKey());
                if (state.activeOptionId != null && !state.activeOptionId.isBlank()) {
                    section.set("active-option", (Object)state.activeOptionId);
                }
                if (state.nextChangeMillis == null || state.nextChangeMillis <= 0L) continue;
                section.set("next-change", (Object)state.nextChangeMillis);
            }
        } else {
            configuration.createSection("rotations");
        }
        try {
            configuration.save(this.stateFile);
        }
        catch (IOException ex) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to save shop rotation state.", ex);
        }
    }

    private static final class RotationState {
        private String activeOptionId;
        private Long nextChangeMillis;

        private RotationState() {
        }

        private RotationState(String activeOptionId, Long nextChangeMillis) {
            this.activeOptionId = activeOptionId;
            this.nextChangeMillis = nextChangeMillis;
        }
    }

    private record UpdateResult(boolean stateChanged, boolean layoutChanged) {
    }
}

