/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.shop;

import com.skyblockexp.common.util.EconomyUtils;
import com.skyblockexp.ezshops.stock.StockMarketManagerHolder;
import com.skyblockexp.shop.DynamicPricingConfiguration;
import com.skyblockexp.shop.ShopMenuLayout;
import com.skyblockexp.shop.ShopPrice;
import com.skyblockexp.shop.ShopPriceType;
import com.skyblockexp.shop.ShopRotationDefinition;
import com.skyblockexp.shop.ShopRotationDurationParser;
import com.skyblockexp.shop.ShopRotationMode;
import com.skyblockexp.shop.ShopRotationOption;
import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Logger;
import org.bukkit.ChatColor;
import org.bukkit.Material;
import org.bukkit.NamespacedKey;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.enchantments.Enchantment;
import org.bukkit.entity.EntityType;
import org.bukkit.plugin.java.JavaPlugin;

public class ShopPricingManager {
    private static final String MAIN_MENU_KEY = "main-menu";
    private static final String CATEGORY_MENU_KEY = "category-menu";
    private static final String CATEGORIES_KEY = "categories";
    private static final String ROTATIONS_KEY = "rotations";
    private final JavaPlugin plugin;
    private final Map<Material, PriceEntry> priceMap = new EnumMap<Material, PriceEntry>(Material.class);
    private final Map<Material, ShopMenuLayout.ItemType> menuItemTypes = new EnumMap<Material, ShopMenuLayout.ItemType>(Material.class);
    private final Logger logger;
    private final File dynamicStateFile;
    private final DynamicPricingConfiguration dynamicConfiguration;
    private YamlConfiguration dynamicStateConfiguration = new YamlConfiguration();
    private ShopMenuLayout menuLayout = ShopMenuLayout.empty();
    private final Map<String, ShopRotationDefinition> rotationDefinitions = new LinkedHashMap<String, ShopRotationDefinition>();
    private final Map<String, String> activeRotationOptions = new LinkedHashMap<String, String>();
    private List<CategoryTemplate> categoryTemplates = List.of();
    private String mainMenuTitle = "Skyblock Shop";
    private int mainMenuSize = 27;
    private ShopMenuLayout.ItemDecoration mainMenuFillDecoration = null;
    private ShopMenuLayout.ItemDecoration defaultBackButtonDecoration = null;
    private int defaultBackButtonSlot = 0;

    public ShopPricingManager(JavaPlugin plugin, DynamicPricingConfiguration dynamicConfiguration) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
        this.dynamicConfiguration = dynamicConfiguration != null ? dynamicConfiguration : DynamicPricingConfiguration.defaults();
        this.dynamicStateFile = new File(plugin.getDataFolder(), "shop-dynamic.yml");
        this.reload();
    }

    public final void reload() {
        this.priceMap.clear();
        this.menuItemTypes.clear();
        this.menuLayout = ShopMenuLayout.empty();
        this.rotationDefinitions.clear();
        this.activeRotationOptions.clear();
        this.categoryTemplates = List.of();
        this.mainMenuTitle = "Skyblock Shop";
        this.mainMenuSize = 27;
        this.mainMenuFillDecoration = null;
        this.defaultBackButtonDecoration = null;
        this.defaultBackButtonSlot = 0;
        this.ensureDataFolder();
        this.dynamicStateConfiguration = this.dynamicStateFile.exists() ? YamlConfiguration.loadConfiguration((File)this.dynamicStateFile) : new YamlConfiguration();
        YamlConfiguration root = this.loadCombinedConfiguration();
        if (root == null) {
            return;
        }
        this.loadLegacyEntries((ConfigurationSection)root);
        this.parseRotations((ConfigurationSection)root);
        this.menuLayout = this.loadMenuLayout((ConfigurationSection)root);
        this.cleanupDynamicState();
    }

    public Optional<ShopPrice> getPrice(Material material) {
        PriceEntry entry = this.priceMap.get(material);
        if (entry == null) {
            return Optional.empty();
        }
        if (entry.priceType == ShopPriceType.STOCK_MARKET) {
            double stockPrice = StockMarketManagerHolder.get().getPrice(material.name());
            return Optional.of(new ShopPrice(stockPrice, stockPrice));
        }
        return Optional.of(entry.currentPrice());
    }

    public boolean isConfigured(Material material) {
        return this.priceMap.containsKey(material);
    }

    public Collection<Material> getBuyableMaterials() {
        return this.filterMaterials(ShopPrice::canBuy);
    }

    public Collection<Material> getSellableMaterials() {
        return this.filterMaterials(ShopPrice::canSell);
    }

    public Set<Material> getConfiguredMaterials() {
        return Collections.unmodifiableSet(this.priceMap.keySet());
    }

    public boolean isEmpty() {
        return this.priceMap.isEmpty();
    }

    public ShopMenuLayout getMenuLayout() {
        return this.menuLayout;
    }

    public Map<String, ShopRotationDefinition> getRotationDefinitions() {
        return Collections.unmodifiableMap(this.rotationDefinitions);
    }

    public Map<String, String> getActiveRotationOptions() {
        return Collections.unmodifiableMap(this.activeRotationOptions);
    }

    public boolean setActiveRotationOption(String rotationId, String optionId) {
        ShopRotationDefinition definition = this.rotationDefinitions.get(rotationId);
        if (definition == null || !definition.containsOption(optionId)) {
            return false;
        }
        String current = this.activeRotationOptions.get(rotationId);
        if (Objects.equals(current, optionId)) {
            return true;
        }
        this.activeRotationOptions.put(rotationId, optionId);
        this.menuLayout = this.rebuildMenuLayoutFromTemplates();
        this.saveRotationState(rotationId, optionId);
        return true;
    }

    public ShopMenuLayout.ItemType getItemType(Material material) {
        if (material == null) {
            return ShopMenuLayout.ItemType.MATERIAL;
        }
        return this.menuItemTypes.getOrDefault(material, ShopMenuLayout.ItemType.MATERIAL);
    }

    private void loadLegacyEntries(ConfigurationSection root) {
        for (String key : root.getKeys(false)) {
            if (MAIN_MENU_KEY.equalsIgnoreCase(key) || CATEGORY_MENU_KEY.equalsIgnoreCase(key) || CATEGORIES_KEY.equalsIgnoreCase(key) || ROTATIONS_KEY.equalsIgnoreCase(key)) continue;
            Material material = Material.matchMaterial((String)key, (boolean)false);
            if (material == null) {
                this.logger.warning("Ignoring unknown material in shop pricing configuration: " + key);
                continue;
            }
            ConfigurationSection priceSection = root.getConfigurationSection(key);
            if (priceSection == null) {
                this.logger.warning("Ignoring price entry for material '" + key + "' because it is not a section.");
                continue;
            }
            double buyPrice = this.readPrice(priceSection, key, "buy");
            double sellPrice = this.readPrice(priceSection, key, "sell");
            if (Double.isNaN(buyPrice) && Double.isNaN(sellPrice)) {
                this.logger.warning("Ignoring price entry for material '" + key + "' because no buy or sell price is defined.");
                continue;
            }
            ShopPrice price = new ShopPrice(Double.isNaN(buyPrice) ? -1.0 : buyPrice, Double.isNaN(sellPrice) ? -1.0 : sellPrice);
            DynamicSettings dynamicSettings = this.parseDynamicSettings(priceSection, key);
            this.registerPrice(material, price, dynamicSettings);
        }
    }

    private void parseRotations(ConfigurationSection root) {
        ConfigurationSection rotationsSection = root.getConfigurationSection(ROTATIONS_KEY);
        if (rotationsSection == null) {
            return;
        }
        ConfigurationSection savedRotations = this.dynamicStateConfiguration != null ? this.dynamicStateConfiguration.getConfigurationSection(ROTATIONS_KEY) : null;
        for (String rotationId : rotationsSection.getKeys(false)) {
            String saved;
            ConfigurationSection rotationSection = rotationsSection.getConfigurationSection(rotationId);
            if (rotationSection == null) {
                this.logger.warning("Ignoring rotation '" + rotationId + "' because it is not a section.");
                continue;
            }
            Duration interval = null;
            String intervalRaw = rotationSection.getString("interval");
            if (intervalRaw != null && !intervalRaw.isBlank() && (interval = ShopRotationDurationParser.parse(intervalRaw)) == null) {
                this.logger.warning("Rotation '" + rotationId + "' has invalid interval '" + intervalRaw + "'.");
            }
            ShopRotationMode mode = ShopRotationMode.fromConfig(rotationSection.getString("mode"));
            ConfigurationSection optionsSection = rotationSection.getConfigurationSection("options");
            if (optionsSection == null || optionsSection.getKeys(false).isEmpty()) {
                this.logger.warning("Ignoring rotation '" + rotationId + "' because it does not define any options.");
                continue;
            }
            ArrayList<ShopRotationOption> options = new ArrayList<ShopRotationOption>();
            for (String optionId : optionsSection.getKeys(false)) {
                double weight;
                ConfigurationSection optionSection = optionsSection.getConfigurationSection(optionId);
                if (optionSection == null) {
                    this.logger.warning("Ignoring option '" + optionId + "' in rotation '" + rotationId + "' because it is not a section.");
                    continue;
                }
                ShopMenuLayout.ItemDecoration iconOverride = this.parseDecoration(optionSection.getConfigurationSection("icon"), null);
                String menuTitleOverride = optionSection.getString("menu-title");
                Map<String, Map<String, Object>> itemOverrides = this.readItemData(optionSection.getConfigurationSection("items"));
                double d = weight = optionSection.contains("weight") ? optionSection.getDouble("weight", 1.0) : 1.0;
                if (weight < 0.0) {
                    this.logger.warning("Rotation option '" + optionId + "' in group '" + rotationId + "' declares a negative weight. Using zero instead.");
                    weight = 0.0;
                }
                options.add(new ShopRotationOption(optionId, iconOverride, menuTitleOverride, itemOverrides, weight));
            }
            if (options.isEmpty()) {
                this.logger.warning("Ignoring rotation '" + rotationId + "' because no valid options were provided.");
                continue;
            }
            String defaultOption = rotationSection.getString("default-option");
            ShopRotationDefinition definition = new ShopRotationDefinition(rotationId, interval, mode, options, defaultOption);
            this.rotationDefinitions.put(rotationId, definition);
            String activeOption = definition.defaultOptionId();
            if (savedRotations != null && definition.containsOption(saved = savedRotations.getString(rotationId))) {
                activeOption = saved;
            }
            this.activeRotationOptions.put(rotationId, activeOption);
        }
    }

    private ShopMenuLayout loadMenuLayout(ConfigurationSection root) {
        ConfigurationSection mainMenuSection = root.getConfigurationSection(MAIN_MENU_KEY);
        this.mainMenuTitle = this.colorize(mainMenuSection != null ? mainMenuSection.getString("title", "&aSkyblock Shop") : "&aSkyblock Shop");
        this.mainMenuSize = this.normalizeSize(mainMenuSection != null ? mainMenuSection.getInt("size", 54) : 54);
        this.mainMenuFillDecoration = this.parseDecoration(mainMenuSection != null ? mainMenuSection.getConfigurationSection("fill") : null, new ShopMenuLayout.ItemDecoration(Material.BLACK_STAINED_GLASS_PANE, 1, String.valueOf(ChatColor.DARK_GRAY) + " ", List.of()));
        ConfigurationSection categoryMenuSection = root.getConfigurationSection(CATEGORY_MENU_KEY);
        this.defaultBackButtonDecoration = this.parseDecoration(categoryMenuSection != null ? categoryMenuSection.getConfigurationSection("back-button") : null, new ShopMenuLayout.ItemDecoration(Material.ARROW, 1, String.valueOf(ChatColor.YELLOW) + "\u2190 Back to Shop", List.of(String.valueOf(ChatColor.GRAY) + "Return to the main shop menu.")));
        this.defaultBackButtonSlot = this.clampSlot(categoryMenuSection != null ? categoryMenuSection.getInt("back-button-slot", 49) : 49, 54);
        ArrayList<CategoryTemplate> templates = new ArrayList<CategoryTemplate>();
        ConfigurationSection categoriesSection = root.getConfigurationSection(CATEGORIES_KEY);
        if (categoriesSection != null) {
            for (String categoryId : categoriesSection.getKeys(false)) {
                ConfigurationSection categorySection = categoriesSection.getConfigurationSection(categoryId);
                if (categorySection == null) {
                    this.logger.warning("Ignoring category '" + categoryId + "' because it is not a section.");
                    continue;
                }
                CategoryTemplate template = this.parseCategoryTemplate(categoryId, categorySection);
                if (template == null) continue;
                templates.add(template);
            }
        }
        this.categoryTemplates = List.copyOf(templates);
        return this.rebuildMenuLayoutFromTemplates();
    }

    private CategoryTemplate parseCategoryTemplate(String categoryId, ConfigurationSection section) {
        String rotationGroupId;
        String displayName = this.colorize(section.getString("name", this.friendlyName(categoryId)));
        ShopMenuLayout.ItemDecoration icon = this.parseDecoration(section.getConfigurationSection("icon"), new ShopMenuLayout.ItemDecoration(Material.CHEST, 1, displayName, List.of()));
        int slot = this.clampSlot(section.getInt("slot", 0), this.mainMenuSize);
        ConfigurationSection menuSection = section.getConfigurationSection("menu");
        String menuTitle = this.colorize(menuSection != null ? menuSection.getString("title", displayName) : displayName);
        int menuSize = this.normalizeSize(menuSection != null ? menuSection.getInt("size", 54) : 54);
        ShopMenuLayout.ItemDecoration menuFill = this.parseDecoration(menuSection != null ? menuSection.getConfigurationSection("fill") : null, null);
        ShopMenuLayout.ItemDecoration backButton = this.parseDecoration(menuSection != null ? menuSection.getConfigurationSection("back-button") : null, null);
        Integer backButtonSlot = null;
        if (menuSection != null && menuSection.contains("back-button-slot")) {
            int slotValue = menuSection.getInt("back-button-slot");
            if (slotValue < 0 || slotValue >= menuSize) {
                this.logger.warning("Ignoring back button slot " + slotValue + " for category '" + categoryId + "' because it is outside the menu bounds.");
            } else {
                backButtonSlot = slotValue;
            }
        }
        if ((rotationGroupId = section.getString("rotation-group")) != null && rotationGroupId.isBlank()) {
            rotationGroupId = null;
        }
        if (rotationGroupId == null) {
            ArrayList<ShopMenuLayout.Item> items = new ArrayList<ShopMenuLayout.Item>();
            ConfigurationSection itemsSection = section.getConfigurationSection("items");
            if (itemsSection == null) {
                this.logger.warning("Category '" + categoryId + "' does not define any items.");
            } else {
                for (String itemId : itemsSection.getKeys(false)) {
                    ConfigurationSection itemSection = itemsSection.getConfigurationSection(itemId);
                    if (itemSection == null) {
                        this.logger.warning("Ignoring item '" + itemId + "' in category '" + categoryId + "' because it is not a section.");
                        continue;
                    }
                    ShopMenuLayout.Item item = this.parseItem("categories." + categoryId + ".items", itemId, itemSection, menuSize);
                    if (item == null) continue;
                    items.add(item);
                }
            }
            String command = section.getString("command", null);
            return new CategoryTemplate(categoryId, displayName, icon, slot, menuTitle, menuSize, menuFill, backButton, backButtonSlot, List.copyOf(items), null, command);
        }
        ShopRotationDefinition definition = this.rotationDefinitions.get(rotationGroupId);
        if (definition == null) {
            this.logger.warning("Category '" + categoryId + "' references unknown rotation-group '" + rotationGroupId + "'.");
            ArrayList<ShopMenuLayout.Item> items = new ArrayList<ShopMenuLayout.Item>();
            ConfigurationSection itemsSection = section.getConfigurationSection("items");
            if (itemsSection != null) {
                for (String itemId : itemsSection.getKeys(false)) {
                    ShopMenuLayout.Item item;
                    ConfigurationSection itemSection = itemsSection.getConfigurationSection(itemId);
                    if (itemSection == null || (item = this.parseItem("categories." + categoryId + ".items", itemId, itemSection, menuSize)) == null) continue;
                    items.add(item);
                }
            }
            String command = section.getString("command", null);
            return new CategoryTemplate(categoryId, displayName, icon, slot, menuTitle, menuSize, menuFill, backButton, backButtonSlot, List.copyOf(items), null, command);
        }
        ConfigurationSection rotationDefaultsSection = section.getConfigurationSection("rotation-defaults");
        ShopMenuLayout.ItemDecoration defaultIcon = this.parseDecoration(rotationDefaultsSection != null ? rotationDefaultsSection.getConfigurationSection("icon") : null, icon);
        String defaultMenuTitle = rotationDefaultsSection != null ? this.colorize(rotationDefaultsSection.getString("menu-title", menuTitle)) : menuTitle;
        Map<String, Map<String, Object>> defaultItemData = this.readItemData(rotationDefaultsSection != null ? rotationDefaultsSection.getConfigurationSection("items") : null);
        if (defaultItemData.isEmpty()) {
            defaultItemData = this.readItemData(section.getConfigurationSection("items"));
        }
        LinkedHashMap<String, List<ShopMenuLayout.Item>> optionItems = new LinkedHashMap<String, List<ShopMenuLayout.Item>>();
        String optionContextPrefixBase = "rotations." + definition.id() + ".options";
        for (ShopRotationOption option : definition.options()) {
            Map<String, Map<String, Object>> mergedItems = this.mergeItemMaps(defaultItemData, option.itemOverrides());
            ArrayList<ShopMenuLayout.Item> parsedItems = new ArrayList<ShopMenuLayout.Item>();
            if (mergedItems.isEmpty()) {
                this.logger.warning("Rotation option '" + option.id() + "' in group '" + definition.id() + "' for category '" + categoryId + "' does not define any items.");
            } else {
                for (Map.Entry<String, Map<String, Object>> entry : mergedItems.entrySet()) {
                    ConfigurationSection itemSection = this.createSectionFromMap(entry.getValue());
                    ShopMenuLayout.Item item = this.parseItem(optionContextPrefixBase + "." + option.id() + ".items", entry.getKey(), itemSection, menuSize);
                    if (item == null) continue;
                    parsedItems.add(item);
                }
            }
            optionItems.put(option.id(), List.copyOf(parsedItems));
        }
        RotationBinding rotation = new RotationBinding(rotationGroupId, defaultIcon, defaultMenuTitle, optionItems);
        return new CategoryTemplate(categoryId, displayName, icon, slot, menuTitle, menuSize, menuFill, backButton, backButtonSlot, List.of(), rotation, section.getString("command", null));
    }

    private ShopMenuLayout rebuildMenuLayoutFromTemplates() {
        ArrayList<ShopMenuLayout.Category> categories = new ArrayList<ShopMenuLayout.Category>(this.categoryTemplates.size());
        for (CategoryTemplate template : this.categoryTemplates) {
            ShopMenuLayout.Category category = this.buildCategoryFromTemplate(template);
            if (category == null) continue;
            categories.add(category);
        }
        return new ShopMenuLayout(this.mainMenuTitle, this.mainMenuSize, this.mainMenuFillDecoration, this.defaultBackButtonDecoration, this.defaultBackButtonSlot, categories);
    }

    private ShopMenuLayout.Category buildCategoryFromTemplate(CategoryTemplate template) {
        List<ShopMenuLayout.Item> items;
        if (template == null) {
            return null;
        }
        if (!template.isRotating()) {
            return new ShopMenuLayout.Category(template.id, template.displayName, template.icon, template.slot, template.menuTitle, template.menuSize, template.menuFill, template.backButton, template.backButtonSlot, template.staticItems, null, template.command);
        }
        RotationBinding binding = template.rotation;
        ShopRotationDefinition definition = this.rotationDefinitions.get(binding.groupId());
        if (definition == null) {
            this.logger.warning("Rotation group '" + binding.groupId() + "' is no longer available for category '" + template.id + "'.");
            return new ShopMenuLayout.Category(template.id, template.displayName, template.icon, template.slot, template.menuTitle, template.menuSize, template.menuFill, template.backButton, template.backButtonSlot, List.of(), null, template.command);
        }
        String optionId = this.activeRotationOptions.getOrDefault(definition.id(), definition.defaultOptionId());
        if (!definition.containsOption(optionId)) {
            optionId = definition.defaultOptionId();
            this.activeRotationOptions.put(definition.id(), optionId);
        }
        ShopRotationOption option = definition.option(optionId).orElse(null);
        ShopMenuLayout.ItemDecoration icon = template.icon;
        if (binding.defaultIcon() != null) {
            icon = binding.defaultIcon();
        }
        if (option != null && option.iconOverride() != null) {
            icon = option.iconOverride();
        }
        String menuTitle = template.menuTitle;
        if (binding.defaultMenuTitle() != null) {
            menuTitle = binding.defaultMenuTitle();
        }
        if (option != null && option.menuTitleOverride() != null) {
            menuTitle = this.colorize(option.menuTitleOverride());
        }
        if ((items = binding.itemsFor(optionId)) == null) {
            items = List.of();
        }
        ShopMenuLayout.CategoryRotation rotationState = new ShopMenuLayout.CategoryRotation(definition.id(), optionId);
        return new ShopMenuLayout.Category(template.id, template.displayName, icon, template.slot, menuTitle, template.menuSize, template.menuFill, template.backButton, template.backButtonSlot, items, rotationState, template.command);
    }

    private ShopMenuLayout.Item parseItem(String contextPrefix, String itemId, ConfigurationSection section, int menuSize) {
        Material material;
        String context = contextPrefix + "." + itemId;
        ShopMenuLayout.ItemType type = ShopMenuLayout.ItemType.fromConfig(section.getString("type"));
        ShopPriceType priceType = ShopPriceType.STATIC;
        String priceTypeStr = section.getString("price-type");
        if (priceTypeStr != null) {
            try {
                priceType = ShopPriceType.valueOf(priceTypeStr.trim().toUpperCase());
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
        }
        if (type == ShopMenuLayout.ItemType.MINION_HEAD) {
            this.logger.warning("Ignoring item '" + context + "' because minion heads are only obtainable from crates.");
            return null;
        }
        String materialKey = section.getString("material", itemId);
        Material material2 = material = materialKey != null ? Material.matchMaterial((String)materialKey, (boolean)false) : null;
        if (material == null) {
            this.logger.warning("Ignoring item '" + context + "' because material '" + materialKey + "' is unknown.");
            return null;
        }
        int slot = section.getInt("slot", -1);
        if (slot < 0 || slot >= menuSize) {
            this.logger.warning("Ignoring item '" + context + "' because slot " + slot + " is outside the menu bounds.");
            return null;
        }
        int amount = Math.max(1, section.getInt("amount", 1));
        int bulkAmount = Math.max(amount, section.getInt("bulk-amount", material.getMaxStackSize()));
        bulkAmount = Math.min(64, bulkAmount);
        double buyPrice = this.readPrice(section, context, "buy");
        double sellPrice = this.readPrice(section, context, "sell");
        if (Double.isNaN(buyPrice) && Double.isNaN(sellPrice)) {
            this.logger.warning("Ignoring item '" + context + "' because no buy or sell price is defined.");
            return null;
        }
        ShopPrice price = new ShopPrice(Double.isNaN(buyPrice) ? -1.0 : buyPrice, Double.isNaN(sellPrice) ? -1.0 : sellPrice);
        DynamicSettings dynamicSettings = this.parseDynamicSettings(section, context);
        if (type == ShopMenuLayout.ItemType.MATERIAL || type == ShopMenuLayout.ItemType.MINION_CRATE_KEY || type == ShopMenuLayout.ItemType.VOTE_CRATE_KEY) {
            this.registerPrice(material, price, dynamicSettings, priceType);
        }
        EntityType spawnerEntity = null;
        if (type == ShopMenuLayout.ItemType.SPAWNER) {
            String entityKey = section.getString("spawner-entity");
            if (entityKey == null || entityKey.isBlank()) {
                this.logger.warning("Ignoring item '" + context + "' because no spawner-entity is provided.");
                return null;
            }
            spawnerEntity = this.parseEntityType(entityKey);
            if (spawnerEntity == null || !spawnerEntity.isSpawnable() || !spawnerEntity.isAlive()) {
                this.logger.warning("Ignoring item '" + context + "' because spawner-entity '" + entityKey + "' is not a valid spawnable entity.");
                return null;
            }
        }
        Map<Object, Object> enchantments = Map.of();
        if (type == ShopMenuLayout.ItemType.ENCHANTED_BOOK) {
            if (material != Material.ENCHANTED_BOOK) {
                this.logger.warning("Ignoring item '" + context + "' because enchanted books must use the ENCHANTED_BOOK material.");
                return null;
            }
            enchantments = this.parseEnchantments(context, section.getConfigurationSection("enchantments"));
            if (enchantments.isEmpty()) {
                this.logger.warning("Ignoring item '" + context + "' because no enchantments are configured for the book.");
                return null;
            }
        }
        Material iconMaterial = material;
        String iconKey = section.getString("icon");
        if (iconKey != null) {
            Material parsedIcon = Material.matchMaterial((String)iconKey, (boolean)false);
            if (parsedIcon == null) {
                this.logger.warning("Unknown icon material '" + iconKey + "' for item '" + context + "'. Using actual material.");
            } else {
                iconMaterial = parsedIcon;
            }
        }
        int iconAmount = Math.max(1, section.getInt("icon-amount", Math.min(amount, 64)));
        iconAmount = Math.min(64, iconAmount);
        String displayName = this.colorize(section.getString("display-name", this.friendlyName(material.name())));
        List<String> lore = this.colorize(section.getStringList("lore"));
        ShopMenuLayout.ItemDecoration decoration = new ShopMenuLayout.ItemDecoration(iconMaterial, iconAmount, displayName, lore);
        int requiredIslandLevel = Math.max(0, section.getInt("required-island-level", 0));
        this.registerMenuItemType(material, type, context);
        return new ShopMenuLayout.Item(itemId, material, decoration, slot, amount, bulkAmount, price, type, spawnerEntity, enchantments, requiredIslandLevel, priceType);
    }

    private Map<String, Map<String, Object>> readItemData(ConfigurationSection section) {
        if (section == null) {
            return Collections.emptyMap();
        }
        LinkedHashMap<String, Map<String, Object>> values = new LinkedHashMap<String, Map<String, Object>>();
        for (String key : section.getKeys(false)) {
            ConfigurationSection child = section.getConfigurationSection(key);
            if (child == null) {
                this.logger.warning("Ignoring item template '" + key + "' because it is not a section.");
                continue;
            }
            values.put(key, this.deepCopyItemData(child));
        }
        return values;
    }

    private Map<String, Map<String, Object>> mergeItemMaps(Map<String, Map<String, Object>> defaults, Map<String, Map<String, Object>> overrides) {
        LinkedHashMap<String, Map<String, Object>> merged = new LinkedHashMap<String, Map<String, Object>>();
        if (defaults != null) {
            for (Map.Entry<String, Map<String, Object>> entry : defaults.entrySet()) {
                merged.put(entry.getKey(), this.deepCopyItemData(entry.getValue()));
            }
        }
        if (overrides != null) {
            for (Map.Entry<String, Map<String, Object>> entry : overrides.entrySet()) {
                Map<String, Object> overrideCopy = this.deepCopyItemData(entry.getValue());
                Map existing = (Map)merged.get(entry.getKey());
                if (existing == null) {
                    merged.put(entry.getKey(), overrideCopy);
                    continue;
                }
                this.applyOverrides(existing, overrideCopy);
            }
        }
        return merged;
    }

    private Map<String, Object> deepCopyItemData(ConfigurationSection section) {
        LinkedHashMap<String, Object> copy = new LinkedHashMap<String, Object>();
        for (String key : section.getKeys(false)) {
            Object value = section.get(key);
            if (value instanceof ConfigurationSection) {
                ConfigurationSection child = (ConfigurationSection)value;
                copy.put(key, this.deepCopyItemData(child));
                continue;
            }
            copy.put(key, value);
        }
        return copy;
    }

    private Map<String, Object> deepCopyItemData(Map<String, Object> source) {
        LinkedHashMap<String, Object> copy = new LinkedHashMap<String, Object>();
        if (source == null) {
            return copy;
        }
        for (Map.Entry<String, Object> entry : source.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map) {
                Map nested = (Map)value;
                copy.put(entry.getKey(), this.deepCopyItemData(nested));
                continue;
            }
            copy.put(entry.getKey(), value);
        }
        return copy;
    }

    private void applyOverrides(Map<String, Object> target, Map<String, Object> overrides) {
        for (Map.Entry<String, Object> entry : overrides.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map) {
                Map nested = (Map)value;
                Object existing = target.get(entry.getKey());
                if (existing instanceof Map) {
                    Map existingMap = (Map)existing;
                    this.applyOverrides(existingMap, nested);
                    continue;
                }
                target.put(entry.getKey(), this.deepCopyItemData(nested));
                continue;
            }
            target.put(entry.getKey(), value);
        }
    }

    private ConfigurationSection createSectionFromMap(Map<String, Object> values) {
        YamlConfiguration configuration = new YamlConfiguration();
        this.populateSection((ConfigurationSection)configuration, values);
        return configuration;
    }

    private void populateSection(ConfigurationSection target, Map<String, Object> values) {
        for (Map.Entry<String, Object> entry : values.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map) {
                Map nested = (Map)value;
                ConfigurationSection child = target.createSection(entry.getKey());
                this.populateSection(child, nested);
                continue;
            }
            target.set(entry.getKey(), value);
        }
    }

    private Collection<Material> filterMaterials(PricePredicate predicate) {
        ArrayList<Material> materials = new ArrayList<Material>();
        for (Map.Entry<Material, PriceEntry> entry : this.priceMap.entrySet()) {
            if (!predicate.test(entry.getValue().currentPrice())) continue;
            materials.add(entry.getKey());
        }
        materials.sort((a, b) -> a.name().compareToIgnoreCase(b.name()));
        return Collections.unmodifiableList(materials);
    }

    private void registerPrice(Material material, ShopPrice price, DynamicSettings dynamicSettings) {
        PriceEntry entry = new PriceEntry(price, dynamicSettings, this.loadSavedMultiplier(material, dynamicSettings));
        PriceEntry previous = this.priceMap.put(material, entry);
        if (previous != null && !previous.basePrice.equals(price)) {
            this.logger.fine("Overriding shop price for material " + material.name() + " with new configuration values.");
        }
    }

    private void registerPrice(Material material, ShopPrice price, DynamicSettings dynamicSettings, ShopPriceType priceType) {
        PriceEntry entry = new PriceEntry(price, dynamicSettings, this.loadSavedMultiplier(material, dynamicSettings), priceType);
        PriceEntry previous = this.priceMap.put(material, entry);
        if (previous != null && !previous.basePrice.equals(price)) {
            this.logger.fine("Overriding shop price for material " + material.name() + " with new configuration values.");
        }
    }

    private void registerMenuItemType(Material material, ShopMenuLayout.ItemType type, String context) {
        ShopMenuLayout.ItemType previous = this.menuItemTypes.get(material);
        if (previous == null) {
            this.menuItemTypes.put(material, type);
            return;
        }
        if (previous == type) {
            return;
        }
        if (previous == ShopMenuLayout.ItemType.MATERIAL && type != ShopMenuLayout.ItemType.MATERIAL) {
            this.menuItemTypes.put(material, type);
            return;
        }
        if (type == ShopMenuLayout.ItemType.MATERIAL) {
            return;
        }
        this.logger.warning("Item '" + context + "' declares type '" + String.valueOf((Object)type) + "' but material '" + material.name() + "' is already registered as '" + String.valueOf((Object)previous) + "'.");
    }

    public void handlePurchase(Material material, int amount) {
        this.adjustDynamicMultiplier(material, amount, true);
    }

    public void handleSale(Material material, int amount) {
        this.adjustDynamicMultiplier(material, amount, false);
    }

    private void adjustDynamicMultiplier(Material material, int amount, boolean purchase) {
        boolean changed;
        if (amount <= 0) {
            return;
        }
        PriceEntry entry = this.priceMap.get(material);
        if (entry == null || !entry.hasDynamicPricing()) {
            return;
        }
        boolean bl = changed = purchase ? entry.adjustAfterPurchase(amount) : entry.adjustAfterSale(amount);
        if (changed) {
            this.saveDynamicState(material, entry);
        }
    }

    private YamlConfiguration loadCombinedConfiguration() {
        File directory;
        File dataFolder = this.plugin.getDataFolder();
        YamlConfiguration combined = new YamlConfiguration();
        File primaryFile = new File(dataFolder, "shop.yml");
        boolean foundConfig = false;
        if (primaryFile.exists()) {
            this.mergeSections((ConfigurationSection)combined, (ConfigurationSection)YamlConfiguration.loadConfiguration((File)primaryFile));
            foundConfig = true;
        }
        if (this.mergeDirectory((ConfigurationSection)combined, directory = new File(dataFolder, "shop"))) {
            foundConfig = true;
        }
        if (!foundConfig) {
            this.logger.warning("Shop pricing file not found: " + primaryFile.getName());
            return null;
        }
        return combined;
    }

    private boolean mergeDirectory(ConfigurationSection target, File directory) {
        if (directory == null || !directory.exists()) {
            return false;
        }
        File[] files = directory.listFiles();
        if (files == null || files.length == 0) {
            return false;
        }
        Arrays.sort(files, Comparator.comparing(File::getName, String::compareToIgnoreCase));
        boolean merged = false;
        for (File file : files) {
            if (file.isDirectory()) {
                merged |= this.mergeDirectory(target, file);
                continue;
            }
            if (!file.getName().toLowerCase(Locale.ROOT).endsWith(".yml")) continue;
            this.mergeSections(target, (ConfigurationSection)YamlConfiguration.loadConfiguration((File)file));
            merged = true;
        }
        return merged;
    }

    private void mergeSections(ConfigurationSection target, ConfigurationSection source) {
        for (String key : source.getKeys(false)) {
            Object value = source.get(key);
            if (value instanceof ConfigurationSection) {
                ConfigurationSection section = (ConfigurationSection)value;
                ConfigurationSection child = target.getConfigurationSection(key);
                if (child == null) {
                    child = target.createSection(key);
                }
                this.mergeSections(child, section);
                continue;
            }
            target.set(key, value);
        }
    }

    private void ensureDataFolder() {
        File dataFolder = this.plugin.getDataFolder();
        if (!dataFolder.exists() && !dataFolder.mkdirs()) {
            this.logger.warning("Unable to create plugin data folder for dynamic shop pricing state.");
        }
    }

    private double loadSavedMultiplier(Material material, DynamicSettings settings) {
        if (settings == null) {
            return 1.0;
        }
        if (this.dynamicStateConfiguration == null) {
            this.dynamicStateConfiguration = new YamlConfiguration();
        }
        if (!this.dynamicStateConfiguration.isSet(material.name())) {
            return settings.clamp(settings.startingMultiplier());
        }
        return settings.clamp(this.dynamicStateConfiguration.getDouble(material.name(), settings.startingMultiplier()));
    }

    private void saveDynamicState(Material material, PriceEntry entry) {
        if (this.dynamicStateConfiguration == null || entry == null || !entry.hasDynamicPricing()) {
            return;
        }
        this.dynamicStateConfiguration.set(material.name(), (Object)entry.multiplier);
        try {
            this.dynamicStateConfiguration.save(this.dynamicStateFile);
        }
        catch (IOException ex) {
            this.logger.warning("Failed to save dynamic shop pricing data: " + ex.getMessage());
        }
    }

    private void saveRotationState(String rotationId, String optionId) {
        ConfigurationSection rotationSection;
        if (this.dynamicStateConfiguration == null) {
            this.dynamicStateConfiguration = new YamlConfiguration();
        }
        if ((rotationSection = this.dynamicStateConfiguration.getConfigurationSection(ROTATIONS_KEY)) == null) {
            rotationSection = this.dynamicStateConfiguration.createSection(ROTATIONS_KEY);
        }
        rotationSection.set(rotationId, (Object)optionId);
        try {
            this.dynamicStateConfiguration.save(this.dynamicStateFile);
        }
        catch (IOException ex) {
            this.logger.warning("Failed to save rotation state: " + ex.getMessage());
        }
    }

    private void cleanupDynamicState() {
        if (this.dynamicStateConfiguration == null) {
            return;
        }
        boolean dirty = false;
        for (Object key : new ArrayList(this.dynamicStateConfiguration.getKeys(false))) {
            PriceEntry entry;
            if (ROTATIONS_KEY.equalsIgnoreCase((String)key)) continue;
            Material material = Material.matchMaterial((String)key, (boolean)false);
            PriceEntry priceEntry = entry = material != null ? this.priceMap.get(material) : null;
            if (material != null && entry != null && entry.hasDynamicPricing()) continue;
            this.dynamicStateConfiguration.set((String)key, null);
            dirty = true;
        }
        ConfigurationSection rotationSection = this.dynamicStateConfiguration.getConfigurationSection(ROTATIONS_KEY);
        if (rotationSection != null) {
            for (String key : new ArrayList(rotationSection.getKeys(false))) {
                ShopRotationDefinition definition = this.rotationDefinitions.get(key);
                String optionId = rotationSection.getString(key);
                if (definition != null && definition.containsOption(optionId)) continue;
                rotationSection.set(key, null);
                dirty = true;
            }
            if (rotationSection.getKeys(false).isEmpty()) {
                this.dynamicStateConfiguration.set(ROTATIONS_KEY, null);
                dirty = true;
            }
        }
        if (dirty) {
            try {
                this.dynamicStateConfiguration.save(this.dynamicStateFile);
            }
            catch (IOException ex) {
                this.logger.warning("Failed to clean up dynamic shop pricing data: " + ex.getMessage());
            }
        }
    }

    private DynamicSettings parseDynamicSettings(ConfigurationSection section, String materialKey) {
        if (section == null || !this.dynamicConfiguration.enabled()) {
            return null;
        }
        ConfigurationSection dynamicSection = section.getConfigurationSection("dynamic-pricing");
        if (dynamicSection == null) {
            return null;
        }
        boolean enabled = dynamicSection.getBoolean("enabled", true);
        if (!enabled) {
            return null;
        }
        double startingMultiplier = this.readDynamicValue(dynamicSection, materialKey, "starting-multiplier", this.dynamicConfiguration.defaultStartingMultiplier());
        double minMultiplier = this.readDynamicValue(dynamicSection, materialKey, "min-multiplier", this.dynamicConfiguration.defaultMinMultiplier());
        double maxMultiplier = this.readDynamicValue(dynamicSection, materialKey, "max-multiplier", this.dynamicConfiguration.defaultMaxMultiplier());
        double buyChange = this.readDynamicValue(dynamicSection, materialKey, "buy-change", this.dynamicConfiguration.defaultBuyChange());
        double sellChange = this.readDynamicValue(dynamicSection, materialKey, "sell-change", this.dynamicConfiguration.defaultSellChange());
        if (Double.isNaN(startingMultiplier) || startingMultiplier <= 0.0) {
            this.logger.warning("Invalid starting-multiplier for dynamic pricing on material '" + materialKey + "'. Using " + this.dynamicConfiguration.defaultStartingMultiplier() + ".");
            startingMultiplier = this.dynamicConfiguration.defaultStartingMultiplier();
        }
        if (Double.isNaN(minMultiplier) || minMultiplier <= 0.0) {
            this.logger.warning("Invalid min-multiplier for dynamic pricing on material '" + materialKey + "'. Using " + this.dynamicConfiguration.defaultMinMultiplier() + ".");
            minMultiplier = this.dynamicConfiguration.defaultMinMultiplier();
        }
        if (Double.isNaN(maxMultiplier) || maxMultiplier <= 0.0) {
            this.logger.warning("Invalid max-multiplier for dynamic pricing on material '" + materialKey + "'. Using " + this.dynamicConfiguration.defaultMaxMultiplier() + ".");
            maxMultiplier = this.dynamicConfiguration.defaultMaxMultiplier();
        }
        if (maxMultiplier < minMultiplier) {
            double temp = maxMultiplier;
            maxMultiplier = minMultiplier;
            minMultiplier = temp;
        }
        if (Double.isNaN(buyChange) || buyChange < 0.0) {
            this.logger.warning("Invalid buy-change for dynamic pricing on material '" + materialKey + "'. Using " + this.dynamicConfiguration.defaultBuyChange() + ".");
            buyChange = this.dynamicConfiguration.defaultBuyChange();
        }
        if (Double.isNaN(sellChange) || sellChange < 0.0) {
            this.logger.warning("Invalid sell-change for dynamic pricing on material '" + materialKey + "'. Using " + this.dynamicConfiguration.defaultSellChange() + ".");
            sellChange = this.dynamicConfiguration.defaultSellChange();
        }
        return new DynamicSettings(startingMultiplier, minMultiplier, maxMultiplier, buyChange, sellChange);
    }

    private double readDynamicValue(ConfigurationSection section, String materialKey, String path, double fallback) {
        if (!section.contains(path)) {
            return fallback;
        }
        Object value = section.get(path);
        if (value instanceof Number) {
            Number number = (Number)value;
            return number.doubleValue();
        }
        if (value instanceof String) {
            String stringValue = (String)value;
            try {
                return Double.parseDouble(stringValue.trim());
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        this.logger.warning("Invalid " + path + " for dynamic pricing on material '" + materialKey + "'. Using " + fallback + ".");
        return fallback;
    }

    private int normalizeSize(int requested) {
        return this.normalizeSize(requested, 54);
    }

    private int normalizeSize(int requested, int defaultSize) {
        int size = requested <= 0 ? defaultSize : requested;
        if ((size = Math.min(54, Math.max(9, size))) % 9 != 0) {
            size += 9 - size % 9;
        }
        return size;
    }

    private int clampSlot(int slot, int menuSize) {
        if (slot < 0) {
            return 0;
        }
        if (slot >= menuSize) {
            return menuSize - 1;
        }
        return slot;
    }

    private ShopMenuLayout.ItemDecoration parseDecoration(ConfigurationSection section, ShopMenuLayout.ItemDecoration fallback) {
        Material material;
        if (section == null) {
            return fallback;
        }
        String materialKey = section.getString("material");
        Material material2 = material = fallback != null ? fallback.material() : Material.AIR;
        if (materialKey != null) {
            Material parsed = Material.matchMaterial((String)materialKey, (boolean)false);
            if (parsed == null) {
                this.logger.warning("Unknown material '" + materialKey + "' in menu decoration configuration.");
            } else {
                material = parsed;
            }
        }
        int amount = Math.max(1, section.getInt("amount", fallback != null ? fallback.amount() : 1));
        String displayName = this.colorize(section.getString("display-name", fallback != null ? fallback.displayName() : null));
        List<String> lore = this.colorize(section.getStringList("lore"));
        return new ShopMenuLayout.ItemDecoration(material, amount, displayName, lore);
    }

    private List<String> colorize(List<String> lines) {
        if (lines == null || lines.isEmpty()) {
            return List.of();
        }
        ArrayList<String> colored = new ArrayList<String>(lines.size());
        for (String line : lines) {
            colored.add(this.colorize(line));
        }
        return colored;
    }

    private String colorize(String text) {
        if (text == null) {
            return null;
        }
        return ChatColor.translateAlternateColorCodes((char)'&', (String)text);
    }

    private String friendlyName(String raw) {
        String lower = raw.toLowerCase(Locale.ENGLISH).replace('_', ' ');
        String[] parts = lower.split(" ");
        StringBuilder builder = new StringBuilder();
        for (String part : parts) {
            if (part.isEmpty()) continue;
            builder.append(Character.toUpperCase(part.charAt(0)));
            if (part.length() > 1) {
                builder.append(part.substring(1));
            }
            builder.append(' ');
        }
        if (builder.length() == 0) {
            return lower;
        }
        builder.setLength(builder.length() - 1);
        return builder.toString();
    }

    private double readPrice(ConfigurationSection section, String materialKey, String path) {
        if (!section.contains(path)) {
            return Double.NaN;
        }
        Object value = section.get(path);
        if (value instanceof Number) {
            Number number = (Number)value;
            return number.doubleValue();
        }
        if (value instanceof String) {
            String stringValue = (String)value;
            try {
                return Double.parseDouble(stringValue.trim());
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        this.logger.warning("Ignoring invalid " + path + " price for material '" + materialKey + "': expected a numeric value.");
        return Double.NaN;
    }

    private Map<Enchantment, Integer> parseEnchantments(String context, ConfigurationSection section) {
        if (section == null) {
            this.logger.warning("Ignoring item '" + context + "' because it does not define any enchantments.");
            return Map.of();
        }
        LinkedHashMap<Enchantment, Integer> values = new LinkedHashMap<Enchantment, Integer>();
        for (String key : section.getKeys(false)) {
            if (key == null) continue;
            Enchantment enchantment = this.parseEnchantment(key);
            if (enchantment == null) {
                this.logger.warning("Ignoring enchantment '" + key + "' for item '" + context + "' because it is not recognized.");
                continue;
            }
            int level = Math.max(1, section.getInt(key, 0));
            if (level <= 0) {
                this.logger.warning("Ignoring enchantment '" + key + "' for item '" + context + "' because the configured level is not positive.");
                continue;
            }
            values.put(enchantment, Math.min(level, enchantment.getMaxLevel()));
        }
        return values;
    }

    private Enchantment parseEnchantment(String key) {
        Enchantment enchantment;
        NamespacedKey namespacedKey;
        if (key == null || key.isBlank()) {
            return null;
        }
        String trimmed = key.trim();
        try {
            namespacedKey = NamespacedKey.fromString((String)trimmed);
        }
        catch (IllegalArgumentException ex) {
            namespacedKey = null;
        }
        if (namespacedKey != null && (enchantment = Enchantment.getByKey((NamespacedKey)namespacedKey)) != null) {
            return enchantment;
        }
        if (!trimmed.contains(":") && (enchantment = Enchantment.getByKey((NamespacedKey)NamespacedKey.minecraft((String)trimmed.toLowerCase(Locale.ROOT)))) != null) {
            return enchantment;
        }
        return Enchantment.getByName((String)trimmed.toUpperCase(Locale.ROOT));
    }

    private EntityType parseEntityType(String value) {
        if (value == null || value.isBlank()) {
            return null;
        }
        String normalized = value.trim().toUpperCase(Locale.ROOT).replace(' ', '_');
        try {
            return EntityType.valueOf((String)normalized);
        }
        catch (IllegalArgumentException ex) {
            return EntityType.fromName((String)value.trim());
        }
    }

    private static final class PriceEntry {
        private final ShopPrice basePrice;
        private final DynamicSettings settings;
        private double multiplier;
        private final ShopPriceType priceType;

        private PriceEntry(ShopPrice basePrice, DynamicSettings settings, double initialMultiplier) {
            this(basePrice, settings, initialMultiplier, ShopPriceType.STATIC);
        }

        private PriceEntry(ShopPrice basePrice, DynamicSettings settings, double initialMultiplier, ShopPriceType priceType) {
            this.basePrice = basePrice;
            this.settings = settings;
            this.multiplier = settings != null ? settings.clamp(initialMultiplier) : 1.0;
            this.priceType = priceType == null ? ShopPriceType.STATIC : priceType;
        }

        private ShopPrice currentPrice() {
            if (!this.hasDynamicPricing()) {
                return this.basePrice;
            }
            double buy = this.basePrice.buyPrice();
            double sell = this.basePrice.sellPrice();
            buy = this.basePrice.canBuy() ? EconomyUtils.normalizeCurrency(buy * this.multiplier) : -1.0;
            sell = this.basePrice.canSell() ? EconomyUtils.normalizeCurrency(sell * this.multiplier) : -1.0;
            return new ShopPrice(buy, sell);
        }

        private boolean hasDynamicPricing() {
            return this.settings != null && this.basePrice != null && (this.basePrice.canBuy() || this.basePrice.canSell());
        }

        private boolean adjustAfterPurchase(int amount) {
            if (!this.hasDynamicPricing() || amount <= 0) {
                return false;
            }
            double change = this.settings.buyChange() * (double)amount;
            return this.applyChange(change);
        }

        private boolean adjustAfterSale(int amount) {
            if (!this.hasDynamicPricing() || amount <= 0) {
                return false;
            }
            double change = -(this.settings.sellChange() * (double)amount);
            return this.applyChange(change);
        }

        private boolean applyChange(double delta) {
            if (delta == 0.0) {
                return false;
            }
            double previous = this.multiplier;
            this.multiplier = this.settings.clamp(this.multiplier + delta);
            return Double.compare(previous, this.multiplier) != 0;
        }
    }

    @FunctionalInterface
    private static interface PricePredicate {
        public boolean test(ShopPrice var1);
    }

    private record DynamicSettings(double startingMultiplier, double minMultiplier, double maxMultiplier, double buyChange, double sellChange) {
        private double clamp(double value) {
            double clamped = Math.min(this.maxMultiplier, Math.max(this.minMultiplier, value));
            return clamped <= 0.0 ? this.minMultiplier : clamped;
        }
    }

    private static final class CategoryTemplate {
        private final String id;
        private final String displayName;
        private final ShopMenuLayout.ItemDecoration icon;
        private final int slot;
        private final String menuTitle;
        private final int menuSize;
        private final ShopMenuLayout.ItemDecoration menuFill;
        private final ShopMenuLayout.ItemDecoration backButton;
        private final Integer backButtonSlot;
        private final List<ShopMenuLayout.Item> staticItems;
        private final RotationBinding rotation;
        private final String command;

        private CategoryTemplate(String id, String displayName, ShopMenuLayout.ItemDecoration icon, int slot, String menuTitle, int menuSize, ShopMenuLayout.ItemDecoration menuFill, ShopMenuLayout.ItemDecoration backButton, Integer backButtonSlot, List<ShopMenuLayout.Item> staticItems, RotationBinding rotation, String command) {
            this.id = Objects.requireNonNull(id, "id");
            this.displayName = Objects.requireNonNull(displayName, "displayName");
            this.icon = icon;
            this.slot = slot;
            this.menuTitle = Objects.requireNonNull(menuTitle, "menuTitle");
            this.menuSize = menuSize;
            this.menuFill = menuFill;
            this.backButton = backButton;
            this.backButtonSlot = backButtonSlot;
            this.staticItems = staticItems == null ? List.of() : List.copyOf(staticItems);
            this.rotation = rotation;
            this.command = command;
        }

        private boolean isRotating() {
            return this.rotation != null;
        }
    }

    private static final class RotationBinding {
        private final String groupId;
        private final ShopMenuLayout.ItemDecoration defaultIcon;
        private final String defaultMenuTitle;
        private final Map<String, List<ShopMenuLayout.Item>> optionItems;

        private RotationBinding(String groupId, ShopMenuLayout.ItemDecoration defaultIcon, String defaultMenuTitle, Map<String, List<ShopMenuLayout.Item>> optionItems) {
            this.groupId = Objects.requireNonNull(groupId, "groupId");
            this.defaultIcon = defaultIcon;
            this.defaultMenuTitle = defaultMenuTitle;
            LinkedHashMap items = new LinkedHashMap();
            if (optionItems != null) {
                for (Map.Entry<String, List<ShopMenuLayout.Item>> entry : optionItems.entrySet()) {
                    items.put(entry.getKey(), entry.getValue() == null ? List.of() : List.copyOf((Collection)entry.getValue()));
                }
            }
            this.optionItems = Collections.unmodifiableMap(items);
        }

        private String groupId() {
            return this.groupId;
        }

        private ShopMenuLayout.ItemDecoration defaultIcon() {
            return this.defaultIcon;
        }

        private String defaultMenuTitle() {
            return this.defaultMenuTitle;
        }

        private List<ShopMenuLayout.Item> itemsFor(String optionId) {
            return this.optionItems.get(optionId);
        }
    }
}

