/*
 * Decompiled with CFR 0.152.
 */
package com.ezflytime.flight;

import com.ezflytime.EzFlyTimePlugin;
import com.ezflytime.flight.FlightBossBarHandler;
import com.ezflytime.storage.FlyTimeStorage;
import com.ezflytime.util.FlightSupport;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import org.bukkit.Bukkit;
import org.bukkit.Material;
import org.bukkit.entity.Player;
import org.bukkit.event.EventHandler;
import org.bukkit.event.Listener;
import org.bukkit.event.player.PlayerJoinEvent;
import org.bukkit.event.player.PlayerMoveEvent;
import org.bukkit.event.player.PlayerToggleFlightEvent;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitRunnable;

public class FlyTimeManager
implements Listener {
    private final EzFlyTimePlugin plugin;
    private final Map<UUID, Integer> remainingSeconds = new HashMap<UUID, Integer>();
    private final Map<UUID, Boolean> activelyFlying = new HashMap<UUID, Boolean>();
    private final FlightBossBarHandler bossBarHandler;
    private BukkitRunnable countdownTask;
    private BukkitRunnable autoSaveTask;
    private final FlyTimeStorage storage;
    private final long autoSaveIntervalTicks;
    private final ActivationMode activationMode;

    public FlyTimeManager(EzFlyTimePlugin plugin, FlyTimeStorage storage) {
        this.plugin = plugin;
        this.storage = storage;
        int autoSaveIntervalSeconds = Math.max(0, plugin.getConfig().getInt("auto-save.interval-seconds", 300));
        this.autoSaveIntervalTicks = (long)autoSaveIntervalSeconds * 20L;
        this.bossBarHandler = new FlightBossBarHandler(plugin);
        this.activationMode = this.parseActivationMode(plugin.getConfig().getString("flight.activation-mode", "NORMAL"));
        this.loadData();
        plugin.getServer().getPluginManager().registerEvents((Listener)this, (Plugin)plugin);
    }

    public void addTime(Player player, int seconds) {
        this.addTime(player, seconds, true);
    }

    public void addTime(Player player, int seconds, boolean notify) {
        UUID uuid = player.getUniqueId();
        int currentTime = this.remainingSeconds.getOrDefault(uuid, 0);
        this.debug("Granting %d seconds of flight time to %s (%s). Previous total: %d seconds.", seconds, player.getName(), uuid, currentTime);
        int newTime = currentTime + seconds;
        this.remainingSeconds.put(uuid, newTime);
        this.debug("Updated flight time for %s (%s): %d seconds total.", player.getName(), uuid, newTime);
        this.bossBarHandler.setInitialFlight(uuid, newTime);
        this.enableFlight(player);
        int minutes = seconds / 60;
        if (notify) {
            player.sendMessage(this.plugin.getMessage("messages.flight-added").replace("{minutes}", String.valueOf(minutes)).replace("{seconds}", String.valueOf(seconds)).replace("{total}", String.valueOf(newTime)));
        }
    }

    public int getRemainingSeconds(Player player) {
        return this.getRemainingSeconds(player.getUniqueId());
    }

    public int getRemainingSeconds(UUID uuid) {
        return this.remainingSeconds.getOrDefault(uuid, 0);
    }

    public Map<UUID, Integer> snapshotRemainingSeconds() {
        return new HashMap<UUID, Integer>(this.remainingSeconds);
    }

    public Map<UUID, Boolean> snapshotActiveStates() {
        return new HashMap<UUID, Boolean>(this.activelyFlying);
    }

    public void restoreState(Map<UUID, Integer> remaining, Map<UUID, Boolean> activeStates) {
        this.remainingSeconds.clear();
        this.remainingSeconds.putAll(remaining);
        this.activelyFlying.clear();
        this.activelyFlying.putAll(activeStates);
        for (Map.Entry<UUID, Integer> entry : remaining.entrySet()) {
            UUID uuid = entry.getKey();
            int seconds = entry.getValue();
            this.bossBarHandler.setInitialFlight(uuid, seconds);
            Player player = Bukkit.getPlayer((UUID)uuid);
            if (player != null) {
                if (this.activelyFlying.getOrDefault(uuid, false).booleanValue() && this.isFlightEngaged(player)) {
                    this.bossBarHandler.markActive(player, seconds, this.plugin.hasBypassUnlimitedFlight(player), this.hasUnlimitedFlight(player));
                    continue;
                }
                this.bossBarHandler.markInactive(player);
                continue;
            }
            this.bossBarHandler.reset(uuid);
        }
    }

    private void enableFlight(Player player) {
        UUID uuid = player.getUniqueId();
        this.debug("Enabling flight for %s (%s). Remaining: %d seconds.", player.getName(), uuid, this.remainingSeconds.getOrDefault(uuid, 0));
        player.setAllowFlight(true);
        if (this.isFlightEngaged(player)) {
            this.markActive(player);
        } else {
            this.markInactive(player);
        }
    }

    private void disableFlight(Player player) {
        if (!player.isOnline()) {
            return;
        }
        UUID uuid = player.getUniqueId();
        this.debug("Disabling flight for %s (%s). Remaining: %d seconds.", player.getName(), uuid, this.remainingSeconds.getOrDefault(uuid, 0));
        if (!this.plugin.hasBypassUnlimitedFlight(player) && !this.hasUnlimitedFlight(player)) {
            player.setFlying(false);
            player.setAllowFlight(false);
            if (this.activationMode == ActivationMode.DOUBLE_JUMP_ELYTRA && FlightSupport.supportsGliding()) {
                FlightSupport.setGliding(player, false);
            }
        }
        this.markInactive(player);
    }

    private boolean hasUnlimitedFlight(Player player) {
        String modeName = player.getGameMode().name();
        return "CREATIVE".equals(modeName) || "SPECTATOR".equals(modeName);
    }

    public void pauseCountdown(Player player) {
        this.markInactive(player);
    }

    public void resumeCountdown(Player player) {
        if (this.isFlightEngaged(player)) {
            this.markActive(player);
        } else {
            this.markInactive(player);
        }
    }

    private void markActive(Player player) {
        UUID uuid = player.getUniqueId();
        this.activelyFlying.put(uuid, true);
        if (!this.remainingSeconds.containsKey(uuid)) {
            return;
        }
        this.bossBarHandler.markActive(player, this.remainingSeconds.getOrDefault(uuid, 0), this.plugin.hasBypassUnlimitedFlight(player), this.hasUnlimitedFlight(player));
    }

    private void markInactive(Player player) {
        this.activelyFlying.put(player.getUniqueId(), false);
        this.bossBarHandler.markInactive(player);
    }

    public void start() {
        if (this.countdownTask != null) {
            return;
        }
        this.countdownTask = new BukkitRunnable(){

            public void run() {
                for (Map.Entry<UUID, Integer> entry : new HashMap<UUID, Integer>(FlyTimeManager.this.remainingSeconds).entrySet()) {
                    UUID uuid = entry.getKey();
                    int seconds = entry.getValue();
                    Player player = Bukkit.getPlayer((UUID)uuid);
                    if (player == null) {
                        FlyTimeManager.this.debug("Countdown tick skipped for offline player %s. Remaining tracked time: %d seconds.", uuid, seconds);
                        FlyTimeManager.this.activelyFlying.remove(uuid);
                        FlyTimeManager.this.bossBarHandler.reset(uuid);
                        if (seconds > 0) continue;
                        FlyTimeManager.this.remainingSeconds.remove(uuid);
                        continue;
                    }
                    if (seconds <= 0) {
                        FlyTimeManager.this.debug("Flight time exhausted for %s (%s). Removing countdown state.", player.getName(), uuid);
                        FlyTimeManager.this.remainingSeconds.remove(uuid);
                        FlyTimeManager.this.activelyFlying.remove(uuid);
                        FlyTimeManager.this.bossBarHandler.reset(uuid);
                        FlyTimeManager.this.disableFlight(player);
                        player.sendMessage(FlyTimeManager.this.plugin.getMessage("messages.flight-ended"));
                        continue;
                    }
                    boolean isActive = FlyTimeManager.this.activelyFlying.getOrDefault(uuid, FlyTimeManager.this.isFlightEngaged(player));
                    if (!isActive || !FlyTimeManager.this.isFlightEngaged(player)) {
                        FlyTimeManager.this.bossBarHandler.update(player, FlyTimeManager.this.remainingSeconds.getOrDefault(uuid, seconds));
                        continue;
                    }
                    int updatedSeconds = seconds - 1;
                    FlyTimeManager.this.debug("Countdown tick for %s (%s): -1 second (from %d to %d).", player.getName(), uuid, seconds, Math.max(updatedSeconds, 0));
                    FlyTimeManager.this.remainingSeconds.put(uuid, updatedSeconds);
                    FlyTimeManager.this.bossBarHandler.update(player, FlyTimeManager.this.remainingSeconds.getOrDefault(uuid, 0));
                }
            }
        };
        this.countdownTask.runTaskTimer((Plugin)this.plugin, 20L, 20L);
        this.startAutoSaveTask();
    }

    public void stop() {
        if (this.countdownTask != null) {
            this.countdownTask.cancel();
            this.countdownTask = null;
        }
        if (this.autoSaveTask != null) {
            this.autoSaveTask.cancel();
            this.autoSaveTask = null;
        }
    }

    private void startAutoSaveTask() {
        if (this.autoSaveIntervalTicks <= 0L || this.autoSaveTask != null) {
            return;
        }
        this.autoSaveTask = new BukkitRunnable(){

            public void run() {
                FlyTimeManager.this.saveData();
            }
        };
        this.autoSaveTask.runTaskTimer((Plugin)this.plugin, this.autoSaveIntervalTicks, this.autoSaveIntervalTicks);
    }

    private void loadData() {
        this.debug("Loading stored flight time data from storage.", new Object[0]);
        Map<UUID, Integer> loaded = this.storage.loadFlyTimes();
        this.debug("Loaded %d flight time entries from storage.", loaded.size());
        loaded.forEach((uuid, seconds) -> this.debug(" - %s has %d seconds stored.", uuid, seconds));
        this.remainingSeconds.putAll(loaded);
    }

    public void saveData() {
        this.debug("Saving %d flight time entries to storage.", this.remainingSeconds.size());
        this.remainingSeconds.forEach((uuid, seconds) -> this.debug(" - %s retaining %d seconds.", uuid, seconds));
        this.storage.saveFlyTimes(this.remainingSeconds);
        this.debug("Flight time data save complete.", new Object[0]);
    }

    @EventHandler
    public void onPlayerJoin(PlayerJoinEvent event) {
        Player player = event.getPlayer();
        if (this.getRemainingSeconds(player) > 0) {
            this.enableFlight(player);
        }
    }

    @EventHandler
    public void onPlayerToggleFlight(PlayerToggleFlightEvent event) {
        Player player = event.getPlayer();
        UUID uuid = player.getUniqueId();
        if (!this.remainingSeconds.containsKey(uuid)) {
            return;
        }
        if (this.activationMode == ActivationMode.DOUBLE_JUMP_ELYTRA && event.isFlying() && this.shouldTriggerElytra(player)) {
            event.setCancelled(true);
            Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
                if (!FlightSupport.isGliding(player)) {
                    FlightSupport.setGliding(player, true);
                }
                this.markActive(player);
            });
            return;
        }
        Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
            if (this.isFlightEngaged(player)) {
                this.markActive(player);
            } else {
                this.markInactive(player);
            }
        });
    }

    @EventHandler
    public void onPlayerMove(PlayerMoveEvent event) {
        Player player = event.getPlayer();
        UUID uuid = player.getUniqueId();
        if (!this.remainingSeconds.containsKey(uuid)) {
            return;
        }
        boolean currentlyActive = this.activelyFlying.getOrDefault(uuid, false);
        boolean isFlying = this.isFlightEngaged(player);
        if (isFlying && !currentlyActive) {
            this.markActive(player);
        } else if (!isFlying && currentlyActive && player.isOnGround()) {
            this.markInactive(player);
        }
    }

    private ActivationMode parseActivationMode(String value) {
        if (value == null) {
            return ActivationMode.NORMAL;
        }
        String normalized = value.trim().toUpperCase(Locale.ROOT).replace('-', '_');
        try {
            ActivationMode mode = ActivationMode.valueOf(normalized);
            if (mode == ActivationMode.DOUBLE_JUMP_ELYTRA && !FlightSupport.supportsElytra()) {
                this.plugin.getLogger().warning("Elytra activation mode is not supported on this server. Using NORMAL.");
                return ActivationMode.NORMAL;
            }
            return mode;
        }
        catch (IllegalArgumentException ex) {
            this.plugin.getLogger().warning("Unknown flight activation mode '" + value + "'. Using NORMAL.");
            return ActivationMode.NORMAL;
        }
    }

    private boolean isFlightEngaged(Player player) {
        if (this.activationMode == ActivationMode.DOUBLE_JUMP_ELYTRA) {
            return player.isFlying() || FlightSupport.isGliding(player);
        }
        return player.isFlying();
    }

    private boolean shouldTriggerElytra(Player player) {
        if (!FlightSupport.supportsGliding()) {
            return false;
        }
        if (!player.isOnGround() && FlightSupport.isGliding(player)) {
            return false;
        }
        if (!this.isElytraEquipped(player)) {
            return false;
        }
        return !player.isFlying();
    }

    private boolean isElytraEquipped(Player player) {
        if (player.getInventory().getChestplate() == null) {
            return false;
        }
        Material elytra = Material.getMaterial((String)"ELYTRA");
        return elytra != null && player.getInventory().getChestplate().getType() == elytra;
    }

    private void debug(String pattern, Object ... args) {
        if (!this.plugin.isDebugEnabled()) {
            return;
        }
        if (args.length == 0) {
            this.plugin.debug(pattern);
        } else {
            this.plugin.debug(String.format(pattern, args));
        }
    }

    private static enum ActivationMode {
        NORMAL,
        DOUBLE_JUMP_ELYTRA;

    }
}

