/*
 * Decompiled with CFR 0.152.
 */
package com.pandadevv.VelocityShield.util;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.pandadevv.VelocityShield.VelocityShield;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class IPCache {
    private final Map<String, CacheEntry> cache;
    private final long cacheDuration;
    private final TimeUnit cacheTimeUnit;
    private final Path cacheFile;
    private final Gson gson;
    private static final int MAX_CACHE_SIZE = 10000;
    private final AtomicInteger currentCacheSize = new AtomicInteger(0);
    private final ScheduledExecutorService cleanupExecutor;
    private static final long CLEANUP_INTERVAL = 5L;
    private static final TimeUnit CLEANUP_TIME_UNIT = TimeUnit.SECONDS;

    public IPCache(long cacheDuration, TimeUnit cacheTimeUnit, Path dataDirectory) {
        this.cache = new ConcurrentHashMap<String, CacheEntry>();
        this.cacheDuration = cacheDuration;
        this.cacheTimeUnit = cacheTimeUnit;
        this.cacheFile = dataDirectory.resolve("ip_cache.json");
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "IPCache-Cleanup");
            t.setDaemon(true);
            return t;
        });
        this.cleanupExecutor.scheduleAtFixedRate(this::cleanExpiredEntries, 5L, 5L, CLEANUP_TIME_UNIT);
        this.loadCache();
    }

    public void cacheResult(String ip, boolean isVPN) {
        if (this.currentCacheSize.get() >= 10000) {
            this.removeOldestEntries(1000);
        }
        long currentTime = System.currentTimeMillis();
        this.cache.put(ip, new CacheEntry(isVPN, currentTime));
        this.currentCacheSize.incrementAndGet();
        this.saveCache();
    }

    public Boolean getCachedResult(String ip) {
        long durationMillis;
        long entryTime;
        CacheEntry entry = this.cache.get(ip);
        if (entry == null) {
            return null;
        }
        long currentTime = System.currentTimeMillis();
        if (currentTime - (entryTime = entry.getTimestamp()) > (durationMillis = this.cacheTimeUnit.toMillis(this.cacheDuration))) {
            this.cache.remove(ip);
            this.currentCacheSize.decrementAndGet();
            this.saveCache();
            return null;
        }
        return entry.isVPN();
    }

    public void clearCache() {
        this.cache.clear();
        this.currentCacheSize.set(0);
        this.saveCache();
    }

    private void loadCache() {
        if (!Files.exists(this.cacheFile, new LinkOption[0])) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(this.cacheFile);){
            Map loadedCache = (Map)this.gson.fromJson((Reader)reader, new TypeToken<Map<String, CacheEntry>>(){}.getType());
            if (loadedCache != null) {
                this.cache.putAll(loadedCache);
                this.currentCacheSize.set(this.cache.size());
                this.cleanExpiredEntries();
            }
        }
        catch (IOException e) {
            VelocityShield.getInstance().getLogger().error("Failed to load IP cache", (Throwable)e);
        }
    }

    private void saveCache() {
        try {
            this.cleanExpiredEntries();
            try (BufferedWriter writer = Files.newBufferedWriter(this.cacheFile, new OpenOption[0]);){
                this.gson.toJson(this.cache, (Appendable)writer);
            }
        }
        catch (IOException e) {
            VelocityShield.getInstance().getLogger().error("Failed to save IP cache", (Throwable)e);
        }
    }

    private void cleanExpiredEntries() {
        long currentTime = System.currentTimeMillis();
        long durationMillis = this.cacheTimeUnit.toMillis(this.cacheDuration);
        AtomicInteger removedCount = new AtomicInteger(0);
        this.cache.entrySet().removeIf(entry -> {
            boolean expired;
            boolean bl = expired = currentTime - ((CacheEntry)entry.getValue()).getTimestamp() > durationMillis;
            if (expired) {
                this.currentCacheSize.decrementAndGet();
                removedCount.incrementAndGet();
            }
            return expired;
        });
        if (removedCount.get() > 0) {
            this.saveCache();
        }
    }

    private void removeOldestEntries(int count) {
        this.cache.entrySet().stream().sorted((e1, e2) -> Long.compare(((CacheEntry)e1.getValue()).getTimestamp(), ((CacheEntry)e2.getValue()).getTimestamp())).limit(count).forEach(entry -> {
            this.cache.remove(entry.getKey());
            this.currentCacheSize.decrementAndGet();
        });
    }

    public void shutdown() {
        this.cleanupExecutor.shutdown();
        try {
            if (!this.cleanupExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.cleanupExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.cleanupExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private static class CacheEntry {
        private final boolean isVPN;
        private final long timestamp;

        public CacheEntry(boolean isVPN, long timestamp) {
            this.isVPN = isVPN;
            this.timestamp = timestamp;
        }

        public boolean isVPN() {
            return this.isVPN;
        }

        public long getTimestamp() {
            return this.timestamp;
        }
    }
}

