/*
 * Decompiled with CFR 0.152.
 */
package org.maboroshi.partyanimals.manager;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.bukkit.Bukkit;
import org.maboroshi.partyanimals.PartyAnimals;
import org.maboroshi.partyanimals.config.settings.MainConfig;
import org.maboroshi.partyanimals.util.Logger;

public class DatabaseManager {
    private final PartyAnimals plugin;
    private final Logger log;
    private HikariDataSource dataSource;
    private String votesTable;
    private String rewardsTable;
    private String serverDataTable;

    public DatabaseManager(PartyAnimals plugin) {
        this.plugin = plugin;
        this.log = plugin.getPluginLogger();
    }

    public void connect() {
        MainConfig.DatabaseSettings settings = this.plugin.getConfiguration().getMainConfig().database;
        MainConfig.PoolSettings pool = settings.pool;
        this.votesTable = settings.tablePrefix + "votes";
        this.rewardsTable = settings.tablePrefix + "offline_rewards";
        this.serverDataTable = settings.tablePrefix + "server_data";
        HikariConfig config = new HikariConfig();
        config.setPoolName("PartyAnimals-Pool");
        config.setConnectionTimeout((long)pool.connectionTimeout);
        config.setMaximumPoolSize(pool.maximumPoolSize);
        config.setLeakDetectionThreshold((long)pool.leakDetectionThreshold);
        String type = settings.type.toLowerCase();
        if (type.equals("mariadb")) {
            config.setDriverClassName("org.mariadb.jdbc.Driver");
            config.setJdbcUrl("jdbc:mariadb://" + settings.host + ":" + settings.port + "/" + settings.database);
            config.setUsername(settings.username);
            config.setPassword(settings.password);
            this.log.info("Connecting to MariaDB database...");
        } else if (type.equals("mysql")) {
            config.setDriverClassName("com.mysql.cj.jdbc.Driver");
            config.setJdbcUrl("jdbc:mysql://" + settings.host + ":" + settings.port + "/" + settings.database + "?useSSL=false&autoReconnect=true");
            config.setUsername(settings.username);
            config.setPassword(settings.password);
            this.log.info("Connecting to MySQL database...");
        } else {
            config.setDriverClassName("org.sqlite.JDBC");
            config.setJdbcUrl("jdbc:sqlite:" + new File(this.plugin.getDataFolder(), "database.db").getAbsolutePath());
            this.log.info("Connecting to SQLite database...");
        }
        try {
            this.dataSource = new HikariDataSource(config);
            this.initializeTables();
            this.log.info("Successfully connected to the database.");
        }
        catch (Exception e) {
            this.log.error("Failed to connect to database! Please check your config.yml.");
            this.log.error("Error: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void initializeTables() {
        String autoInc = this.isSQLite() ? "AUTOINCREMENT" : "AUTO_INCREMENT";
        String createVotesTable = "CREATE TABLE IF NOT EXISTS " + this.votesTable + " (id INTEGER PRIMARY KEY " + autoInc + ", uuid VARCHAR(36) NOT NULL, username VARCHAR(16) NOT NULL, amount INTEGER NOT NULL DEFAULT 1, service VARCHAR(64) NOT NULL, timestamp LONG NOT NULL);";
        String createRewardsTable = "CREATE TABLE IF NOT EXISTS " + this.rewardsTable + " (id INTEGER PRIMARY KEY " + autoInc + ", uuid VARCHAR(36) NOT NULL, command TEXT NOT NULL);";
        String createServerDataTable = "CREATE TABLE IF NOT EXISTS " + this.serverDataTable + " (key VARCHAR(64) PRIMARY KEY, value TEXT);";
        String indexName = this.votesTable + "_uuid_idx";
        String createIndex = "CREATE INDEX IF NOT EXISTS " + indexName + " ON " + this.votesTable + "(uuid);";
        try (Connection connection = this.getConnection();
             Statement statement = connection.createStatement();){
            statement.execute(createVotesTable);
            statement.execute(createRewardsTable);
            statement.execute(createServerDataTable);
            statement.execute(createIndex);
            this.log.info("Database tables initialized (" + this.votesTable + ", " + this.rewardsTable + ", " + this.serverDataTable + ").");
        }
        catch (SQLException e) {
            this.log.error("Failed to create database tables: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public void addVote(UUID uuid, String username, String service, int amount) {
        String sql = "INSERT INTO " + this.votesTable + " (uuid, username, service, amount, timestamp) VALUES (?, ?, ?, ?, ?);";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setString(1, uuid.toString());
            statement.setString(2, username);
            statement.setString(3, service);
            statement.setInt(4, amount);
            statement.setLong(5, System.currentTimeMillis());
            statement.executeUpdate();
            this.log.debug("Saved vote for " + username);
        }
        catch (SQLException e) {
            this.log.error("Failed to save vote: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getVotes(UUID uuid) {
        String sql = "SELECT SUM(amount) FROM " + this.votesTable + " WHERE uuid = ?;";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setString(1, uuid.toString());
            ResultSet resultSet = statement.executeQuery();
            if (!resultSet.next()) return 0;
            int n = resultSet.getInt(1);
            return n;
        }
        catch (SQLException e) {
            this.log.error("Failed to get votes: " + e.getMessage());
            e.printStackTrace();
        }
        return 0;
    }

    public UUID getPlayerUUID(String playerName) {
        boolean forceOffline = this.plugin.getConfiguration().getMainConfig().modules.vote.forceOfflineMode;
        if (forceOffline) {
            return UUID.nameUUIDFromBytes(("OfflinePlayer:" + playerName).getBytes(StandardCharsets.UTF_8));
        }
        try {
            return Bukkit.createProfile((String)playerName).getId();
        }
        catch (Exception exception) {
            return Bukkit.getOfflinePlayer((String)playerName).getUniqueId();
        }
    }

    public void queueRewards(UUID uuid, String command) {
        String sql = "INSERT INTO " + this.rewardsTable + " (uuid, command) VALUES (?, ?);";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setString(1, uuid.toString());
            statement.setString(2, command);
            statement.executeUpdate();
            this.log.info("Queued reward for " + String.valueOf(uuid));
        }
        catch (SQLException e) {
            this.log.error("Failed to queue reward: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public List<String> retrieveRewards(UUID uuid) {
        ArrayList<String> commands;
        block28: {
            commands = new ArrayList<String>();
            String selectSql = "SELECT command FROM " + this.rewardsTable + " WHERE uuid = ?;";
            String deleteSql = "DELETE FROM " + this.rewardsTable + " WHERE uuid = ?;";
            try {
                try (Connection connection = this.getConnection();
                     PreparedStatement selectStatement = connection.prepareStatement(selectSql);){
                    selectStatement.setString(1, uuid.toString());
                    ResultSet resultSet = selectStatement.executeQuery();
                    while (resultSet.next()) {
                        commands.add(resultSet.getString("command"));
                    }
                }
                if (commands.isEmpty()) break block28;
                connection = this.getConnection();
                try (PreparedStatement deleteStatement = connection.prepareStatement(deleteSql);){
                    deleteStatement.setString(1, uuid.toString());
                    deleteStatement.executeUpdate();
                }
                finally {
                    if (connection != null) {
                        connection.close();
                    }
                }
            }
            catch (SQLException e) {
                this.log.error("Failed to retrieve rewards: " + e.getMessage());
                e.printStackTrace();
            }
        }
        return commands;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getCommunityGoalProgress() {
        String sql = "SELECT value FROM " + this.serverDataTable + " WHERE key = 'community_vote_count';";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            ResultSet rs = statement.executeQuery();
            if (!rs.next()) return 0;
            int n = Integer.parseInt(rs.getString("value"));
            return n;
        }
        catch (Exception exception) {
            // empty catch block
        }
        return 0;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int incrementCommunityGoalProgress() {
        String updateSql = "UPDATE " + this.serverDataTable + " SET value = value + 1 WHERE key = 'community_vote_count';";
        String selectSql = "SELECT value FROM " + this.serverDataTable + " WHERE key = 'community_vote_count';";
        try (Connection connection = this.getConnection();){
            try (PreparedStatement updateStmt = connection.prepareStatement(updateSql);){
                int rowsAffected = updateStmt.executeUpdate();
                if (rowsAffected == 0) {
                    this.setCommunityGoalProgress(1);
                    int n = 1;
                    return n;
                }
            }
            try (PreparedStatement selectStmt = connection.prepareStatement(selectSql);){
                ResultSet rs = selectStmt.executeQuery();
                if (!rs.next()) return 0;
                int n = Integer.parseInt(rs.getString("value"));
                return n;
            }
        }
        catch (SQLException e) {
            this.log.error("Failed to atomic increment community goal: " + e.getMessage());
            e.printStackTrace();
        }
        return 0;
    }

    private void setCommunityGoalProgress(int value) {
        String sql = "REPLACE INTO " + this.serverDataTable + " (key, value) VALUES ('community_vote_count', ?);";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setString(1, String.valueOf(value));
            statement.executeUpdate();
        }
        catch (SQLException e) {
            this.log.error("Failed to update community goal: " + e.getMessage());
        }
    }

    public List<TopVoter> getTopVoters(long afterTimestamp, int limit) {
        ArrayList<TopVoter> results = new ArrayList<TopVoter>();
        String sql = "SELECT MAX(username) as username, SUM(amount) as total FROM " + this.votesTable + " WHERE timestamp >= ? GROUP BY uuid ORDER BY total DESC LIMIT ?;";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setLong(1, afterTimestamp);
            statement.setInt(2, limit);
            ResultSet rs = statement.executeQuery();
            while (rs.next()) {
                results.add(new TopVoter(rs.getString("username"), rs.getInt("total")));
            }
        }
        catch (SQLException e) {
            this.log.error("Failed to calculate top voters: " + e.getMessage());
            e.printStackTrace();
        }
        return results;
    }

    public void disconnect() {
        if (this.dataSource != null && !this.dataSource.isClosed()) {
            this.dataSource.close();
            this.log.info("Database connection pool closed.");
        }
    }

    public Connection getConnection() throws SQLException {
        return this.dataSource.getConnection();
    }

    private boolean isSQLite() {
        if (this.dataSource == null) {
            return false;
        }
        String driver = this.dataSource.getDriverClassName();
        return driver != null && driver.contains("sqlite");
    }

    public record TopVoter(String name, int votes) {
    }
}

