/*
 * Decompiled with CFR 0.152.
 */
package com.jotrorox.shop.database;

import com.jotrorox.shop.model.Shop;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.java.JavaPlugin;

public class DatabaseManager {
    private final JavaPlugin plugin;
    private Connection connection;
    private final ExecutorService executor;

    public DatabaseManager(JavaPlugin plugin) {
        this.plugin = plugin;
        this.executor = Executors.newSingleThreadExecutor();
        this.initDatabase();
    }

    private void initDatabase() {
        try {
            File dataFolder = this.plugin.getDataFolder();
            if (!dataFolder.exists()) {
                dataFolder.mkdirs();
            }
            File dbFile = new File(dataFolder, "shops.db");
            this.connection = DriverManager.getConnection("jdbc:sqlite:" + dbFile.getAbsolutePath());
            try (Statement statement = this.connection.createStatement();){
                statement.execute("PRAGMA journal_mode=WAL;");
                statement.execute("PRAGMA synchronous=NORMAL;");
                statement.execute("CREATE TABLE IF NOT EXISTS shops (sign_location TEXT PRIMARY KEY,chest_location TEXT NOT NULL,owner TEXT NOT NULL,sell_item TEXT,payment_item TEXT,take_amount INTEGER,output_amount INTEGER,shop_name TEXT,show_display BOOLEAN,show_stock_message BOOLEAN,display_entity_id TEXT);");
            }
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Could not initialize SQLite database!");
            e.printStackTrace();
        }
    }

    public void closeConnection() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
            }
            this.executor.shutdown();
        }
        catch (SQLException e) {
            e.printStackTrace();
        }
    }

    private String serializeLocation(Location loc) {
        if (loc == null) {
            return null;
        }
        return loc.getWorld().getName() + ":" + loc.getX() + ":" + loc.getY() + ":" + loc.getZ() + ":" + loc.getYaw() + ":" + loc.getPitch();
    }

    private Location deserializeLocation(String s) {
        if (s == null) {
            return null;
        }
        String[] parts = s.split(":");
        if (parts.length < 6) {
            return null;
        }
        World world = Bukkit.getWorld((String)parts[0]);
        if (world == null) {
            return null;
        }
        return new Location(world, Double.parseDouble(parts[1]), Double.parseDouble(parts[2]), Double.parseDouble(parts[3]), Float.parseFloat(parts[4]), Float.parseFloat(parts[5]));
    }

    private String serializeItemStack(ItemStack item) {
        if (item == null) {
            return null;
        }
        YamlConfiguration config = new YamlConfiguration();
        config.set("item", (Object)item);
        return config.saveToString();
    }

    private ItemStack deserializeItemStack(String s) {
        if (s == null) {
            return null;
        }
        YamlConfiguration config = new YamlConfiguration();
        try {
            config.loadFromString(s);
            return config.getItemStack("item");
        }
        catch (Exception e) {
            return null;
        }
    }

    public CompletableFuture<Void> saveShop(Shop shop) {
        if (this.connection == null) {
            return CompletableFuture.completedFuture(null);
        }
        return CompletableFuture.runAsync(() -> {
            String sql = "INSERT OR REPLACE INTO shops (sign_location, chest_location, owner, sell_item, payment_item, take_amount, output_amount, shop_name, show_display, show_stock_message, display_entity_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                pstmt.setString(1, this.serializeLocation(shop.getSignLocation()));
                pstmt.setString(2, this.serializeLocation(shop.getChestLocation()));
                pstmt.setString(3, shop.getOwner().toString());
                pstmt.setString(4, this.serializeItemStack(shop.getSellItem()));
                pstmt.setString(5, this.serializeItemStack(shop.getPaymentItem()));
                pstmt.setInt(6, shop.getTakeAmount());
                pstmt.setInt(7, shop.getOutputAmount());
                pstmt.setString(8, shop.getShopName());
                pstmt.setBoolean(9, shop.isShowDisplay());
                pstmt.setBoolean(10, shop.isShowStockMessage());
                pstmt.setString(11, shop.getDisplayEntityId() != null ? shop.getDisplayEntityId().toString() : null);
                pstmt.executeUpdate();
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Could not save shop to SQLite!");
                e.printStackTrace();
            }
        }, this.executor);
    }

    public CompletableFuture<Void> removeShop(Location signLoc) {
        if (this.connection == null) {
            return CompletableFuture.completedFuture(null);
        }
        return CompletableFuture.runAsync(() -> {
            String sql = "DELETE FROM shops WHERE sign_location = ?";
            try (PreparedStatement pstmt = this.connection.prepareStatement(sql);){
                pstmt.setString(1, this.serializeLocation(signLoc));
                pstmt.executeUpdate();
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Could not remove shop from SQLite!");
                e.printStackTrace();
            }
        }, this.executor);
    }

    public CompletableFuture<List<Shop>> loadShops() {
        if (this.connection == null) {
            return CompletableFuture.completedFuture(new ArrayList());
        }
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<Shop> shops = new ArrayList<Shop>();
            String sql = "SELECT * FROM shops";
            try (Statement stmt = this.connection.createStatement();
                 ResultSet rs = stmt.executeQuery(sql);){
                while (rs.next()) {
                    Location signLoc = this.deserializeLocation(rs.getString("sign_location"));
                    Location chestLoc = this.deserializeLocation(rs.getString("chest_location"));
                    if (signLoc == null || chestLoc == null) continue;
                    UUID owner = UUID.fromString(rs.getString("owner"));
                    Shop shop = new Shop(signLoc, chestLoc, owner);
                    shop.setSellItem(this.deserializeItemStack(rs.getString("sell_item")));
                    shop.setPaymentItem(this.deserializeItemStack(rs.getString("payment_item")));
                    shop.setTakeAmount(rs.getInt("take_amount"));
                    shop.setOutputAmount(rs.getInt("output_amount"));
                    shop.setShopName(rs.getString("shop_name"));
                    shop.setShowDisplay(rs.getBoolean("show_display"));
                    shop.setShowStockMessage(rs.getBoolean("show_stock_message"));
                    String displayId = rs.getString("display_entity_id");
                    if (displayId != null) {
                        shop.setDisplayEntityId(UUID.fromString(displayId));
                    }
                    shops.add(shop);
                }
            }
            catch (SQLException e) {
                this.plugin.getLogger().severe("Could not load shops from SQLite!");
                e.printStackTrace();
            }
            return shops;
        }, this.executor);
    }
}

