/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.jextranslate.validation;

import de.jexcellence.jextranslate.R18nManager;
import de.jexcellence.jextranslate.config.R18nConfiguration;
import de.jexcellence.jextranslate.validation.ValidationReport;
import de.jexcellence.jextranslate.validation.ValidationStatistics;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.kyori.adventure.text.minimessage.MiniMessage;
import org.jetbrains.annotations.NotNull;

public final class KeyValidator {
    private static final Logger LOGGER = Logger.getLogger(KeyValidator.class.getName());
    private static final MiniMessage MINI_MESSAGE = MiniMessage.miniMessage();
    private static final Pattern PLACEHOLDER_PATTERN = Pattern.compile("\\{([^}]+)}|%([^%]+)%");
    private static final Pattern VALID_KEY_PATTERN = Pattern.compile("^[a-z][a-z0-9._]*$");
    private final R18nConfiguration configuration;

    public KeyValidator(@NotNull R18nConfiguration configuration) {
        this.configuration = configuration;
    }

    @NotNull
    public ValidationReport validateAllKeys() {
        long startTime = System.currentTimeMillis();
        LOGGER.log(Level.INFO, "Starting comprehensive key validation...");
        ValidationReport.Builder reportBuilder = ValidationReport.builder().timestamp(Instant.now());
        Map<String, Map<String, List<String>>> translations = this.getTranslations();
        if (translations.isEmpty()) {
            LOGGER.log(Level.WARNING, "No translations loaded - skipping validation");
            return reportBuilder.statistics(ValidationStatistics.empty()).build();
        }
        String defaultLocale = this.configuration.defaultLocale();
        Set<String> supportedLocales = this.configuration.supportedLocales();
        Set<String> allKeys = translations.keySet();
        this.validateMissingKeys(translations, defaultLocale, supportedLocales, reportBuilder);
        this.validatePlaceholderConsistency(translations, defaultLocale, supportedLocales, reportBuilder);
        this.validateMiniMessageFormat(translations, reportBuilder);
        this.validateNamingConventions(allKeys, reportBuilder);
        ValidationStatistics statistics = this.calculateStatistics(translations, supportedLocales, startTime);
        reportBuilder.statistics(statistics);
        ValidationReport report = reportBuilder.build();
        this.logValidationResults(report);
        return report;
    }

    private void validateMissingKeys(@NotNull Map<String, Map<String, List<String>>> translations, @NotNull String defaultLocale, @NotNull Set<String> supportedLocales, @NotNull ValidationReport.Builder reportBuilder) {
        for (Map.Entry<String, Map<String, List<String>>> entry : translations.entrySet()) {
            String key = entry.getKey();
            Map<String, List<String>> localeMap = entry.getValue();
            for (String locale : supportedLocales) {
                boolean isMissing;
                List<String> messages = localeMap.get(locale);
                boolean bl = isMissing = messages == null || messages.isEmpty();
                if (isMissing && !locale.equals(defaultLocale)) {
                    List<String> defaultMessages = localeMap.get(defaultLocale);
                    if (defaultMessages == null || defaultMessages.isEmpty()) continue;
                    reportBuilder.addMissingKey(key + " [" + locale + "]");
                    continue;
                }
                if (!isMissing || !locale.equals(defaultLocale)) continue;
                reportBuilder.addMissingKey(key + " [DEFAULT:" + defaultLocale + "]");
            }
        }
    }

    private void validatePlaceholderConsistency(@NotNull Map<String, Map<String, List<String>>> translations, @NotNull String defaultLocale, @NotNull Set<String> supportedLocales, @NotNull ValidationReport.Builder reportBuilder) {
        for (Map.Entry<String, Map<String, List<String>>> entry : translations.entrySet()) {
            String key = entry.getKey();
            Map<String, List<String>> localeMap = entry.getValue();
            Set<String> defaultPlaceholders = this.extractPlaceholders(localeMap.get(defaultLocale));
            if (defaultPlaceholders.isEmpty()) continue;
            for (String locale : supportedLocales) {
                List<String> messages;
                if (locale.equals(defaultLocale) || (messages = localeMap.get(locale)) == null || messages.isEmpty()) continue;
                Set<String> localePlaceholders = this.extractPlaceholders(messages);
                HashSet<String> missingInLocale = new HashSet<String>(defaultPlaceholders);
                missingInLocale.removeAll(localePlaceholders);
                HashSet<String> extraInLocale = new HashSet<String>(localePlaceholders);
                extraInLocale.removeAll(defaultPlaceholders);
                if (missingInLocale.isEmpty() && extraInLocale.isEmpty()) continue;
                StringBuilder issue = new StringBuilder(key).append(" [").append(locale).append("]");
                if (!missingInLocale.isEmpty()) {
                    issue.append(" missing: ").append(missingInLocale);
                }
                if (!extraInLocale.isEmpty()) {
                    issue.append(" extra: ").append(extraInLocale);
                }
                reportBuilder.addPlaceholderIssue(issue.toString());
            }
        }
    }

    private void validateMiniMessageFormat(@NotNull Map<String, Map<String, List<String>>> translations, @NotNull ValidationReport.Builder reportBuilder) {
        for (Map.Entry<String, Map<String, List<String>>> entry : translations.entrySet()) {
            String key = entry.getKey();
            Map<String, List<String>> localeMap = entry.getValue();
            for (Map.Entry<String, List<String>> localeEntry : localeMap.entrySet()) {
                String locale = localeEntry.getKey();
                List<String> messages = localeEntry.getValue();
                if (messages == null) continue;
                for (String message : messages) {
                    if (this.isValidMiniMessage(message)) continue;
                    reportBuilder.addFormatError(key + " [" + locale + "]: Invalid MiniMessage format");
                }
            }
        }
    }

    private void validateNamingConventions(@NotNull Set<String> allKeys, @NotNull ValidationReport.Builder reportBuilder) {
        for (String key : allKeys) {
            if (VALID_KEY_PATTERN.matcher(key).matches()) continue;
            if (key.contains(" ")) {
                reportBuilder.addNamingViolation(key + ": Contains spaces");
                continue;
            }
            if (key.matches(".*[A-Z].*")) {
                reportBuilder.addNamingViolation(key + ": Contains uppercase letters");
                continue;
            }
            if (key.startsWith(".") || key.endsWith(".")) {
                reportBuilder.addNamingViolation(key + ": Invalid dot placement");
                continue;
            }
            if (!key.contains("..")) continue;
            reportBuilder.addNamingViolation(key + ": Contains consecutive dots");
        }
    }

    @NotNull
    private Set<String> extractPlaceholders(List<String> messages) {
        HashSet<String> placeholders = new HashSet<String>();
        if (messages == null) {
            return placeholders;
        }
        for (String message : messages) {
            Matcher matcher = PLACEHOLDER_PATTERN.matcher(message);
            while (matcher.find()) {
                String placeholder = matcher.group(1) != null ? matcher.group(1) : matcher.group(2);
                if (placeholder == null) continue;
                placeholders.add(placeholder);
            }
        }
        return placeholders;
    }

    private boolean isValidMiniMessage(@NotNull String message) {
        try {
            String testMessage = message.replaceAll("\\{[^}]+}", "test").replaceAll("%[^%]+%", "test");
            MINI_MESSAGE.deserialize((Object)testMessage);
            return true;
        }
        catch (Exception e) {
            if (this.configuration.debugMode()) {
                LOGGER.log(Level.FINE, "MiniMessage parse error: " + e.getMessage());
            }
            return false;
        }
    }

    @NotNull
    private ValidationStatistics calculateStatistics(@NotNull Map<String, Map<String, List<String>>> translations, @NotNull Set<String> supportedLocales, long startTime) {
        int totalKeys = translations.size();
        int totalLocales = supportedLocales.size();
        HashMap<String, Integer> keysPerLocale = new HashMap<String, Integer>();
        int totalTranslations = 0;
        int expectedTranslations = totalKeys * totalLocales;
        for (String locale : supportedLocales) {
            int count = 0;
            for (Map<String, List<String>> localeMap : translations.values()) {
                List<String> messages = localeMap.get(locale);
                if (messages == null || messages.isEmpty()) continue;
                ++count;
            }
            keysPerLocale.put(locale, count);
            totalTranslations += count;
        }
        double completeness = expectedTranslations > 0 ? (double)totalTranslations * 100.0 / (double)expectedTranslations : 100.0;
        long duration = System.currentTimeMillis() - startTime;
        return new ValidationStatistics(totalKeys, totalLocales, keysPerLocale, completeness, duration);
    }

    private void logValidationResults(@NotNull ValidationReport report) {
        if (report.hasIssues()) {
            LOGGER.warning(String.format("Validation completed with %d issues: %d missing keys, %d format errors, %d placeholder issues, %d naming violations", report.getTotalIssues(), report.missingKeys().size(), report.formatErrors().size(), report.placeholderIssues().size(), report.namingViolations().size()));
            LOGGER.info("Validation Score: " + String.format("%.1f%%", report.getValidationScore()));
        } else {
            LOGGER.info("Validation completed successfully - no issues found!");
            LOGGER.info("Completeness: " + String.format("%.1f%%", report.statistics().completenessPercentage()));
        }
    }

    @NotNull
    private Map<String, Map<String, List<String>>> getTranslations() {
        try {
            R18nManager manager = R18nManager.getInstance();
            if (manager != null) {
                return manager.getTranslationLoader().getAllTranslations();
            }
            return Collections.emptyMap();
        }
        catch (Exception e) {
            LOGGER.warning("Could not access R18nManager translations: " + e.getMessage());
            return Collections.emptyMap();
        }
    }

    public boolean isKeyValid(@NotNull String key, @NotNull String locale) {
        Map<String, Map<String, List<String>>> translations = this.getTranslations();
        Map<String, List<String>> localeMap = translations.get(key);
        if (localeMap == null) {
            return false;
        }
        List<String> messages = localeMap.get(locale);
        if (messages == null || messages.isEmpty()) {
            messages = localeMap.get(this.configuration.defaultLocale());
        }
        return messages != null && !messages.isEmpty();
    }

    @NotNull
    public Set<String> getMissingKeysForLocale(@NotNull String locale) {
        HashSet<String> missingKeys = new HashSet<String>();
        Map<String, Map<String, List<String>>> translations = this.getTranslations();
        String defaultLocale = this.configuration.defaultLocale();
        for (Map.Entry<String, Map<String, List<String>>> entry : translations.entrySet()) {
            boolean existsInDefault;
            String key = entry.getKey();
            Map<String, List<String>> localeMap = entry.getValue();
            List<String> messages = localeMap.get(locale);
            List<String> defaultMessages = localeMap.get(defaultLocale);
            boolean missingInLocale = messages == null || messages.isEmpty();
            boolean bl = existsInDefault = defaultMessages != null && !defaultMessages.isEmpty();
            if (!missingInLocale || !existsInDefault || locale.equals(defaultLocale)) continue;
            missingKeys.add(key);
        }
        return missingKeys;
    }

    @NotNull
    public R18nConfiguration getConfiguration() {
        return this.configuration;
    }
}

