/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.jextranslate.core;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.jexcellence.jextranslate.config.R18nConfiguration;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.CodeSource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.yaml.snakeyaml.Yaml;

public final class TranslationLoader {
    private static final Logger LOGGER = Logger.getLogger(TranslationLoader.class.getName());
    private final JavaPlugin plugin;
    private final R18nConfiguration configuration;
    private final Yaml yaml;
    private final ObjectMapper jsonMapper;
    private final Map<String, Map<String, List<String>>> translations = new ConcurrentHashMap<String, Map<String, List<String>>>();
    private final Set<String> loadedLocales = ConcurrentHashMap.newKeySet();
    private final Set<String> yamlLocales = ConcurrentHashMap.newKeySet();

    public TranslationLoader(@NotNull JavaPlugin plugin, @NotNull R18nConfiguration configuration) {
        this.plugin = plugin;
        this.configuration = configuration;
        this.yaml = new Yaml();
        this.jsonMapper = new ObjectMapper();
    }

    @NotNull
    public FileType detectFileType(@NotNull Path filePath) {
        String fileName = filePath.getFileName().toString().toLowerCase();
        if (fileName.endsWith(".json")) {
            return FileType.JSON;
        }
        if (fileName.endsWith(".yml") || fileName.endsWith(".yaml")) {
            return FileType.YAML;
        }
        return FileType.UNKNOWN;
    }

    @NotNull
    public CompletableFuture<Void> loadTranslations() {
        return CompletableFuture.runAsync(() -> {
            try {
                LOGGER.info("Loading translations for plugin: " + this.plugin.getName());
                LOGGER.info("Plugin class: " + this.plugin.getClass().getName());
                LOGGER.info("Plugin data folder: " + this.plugin.getDataFolder().getAbsolutePath());
                this.translations.clear();
                this.loadedLocales.clear();
                File translationDir = this.getTranslationDirectory();
                LOGGER.info("Translation directory: " + translationDir.getAbsolutePath());
                if (!this.plugin.getDataFolder().exists()) {
                    boolean created = this.plugin.getDataFolder().mkdirs();
                    LOGGER.info("Created plugin data folder: " + created);
                }
                this.ensureDirectoryExists(translationDir);
                LOGGER.info("Translation directory exists: " + translationDir.exists());
                String testResource = this.configuration.translationDirectory() + "/en_US.yml";
                InputStream testStream = this.plugin.getResource(testResource);
                LOGGER.info("Test resource '" + testResource + "' exists: " + (testStream != null));
                if (testStream != null) {
                    try {
                        testStream.close();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
                this.extractResourceFiles(translationDir);
                File[] files = translationDir.listFiles();
                if (files != null) {
                    LOGGER.info("Files in translation directory: " + files.length);
                    for (File file : files) {
                        LOGGER.info("  - " + file.getName() + " (" + file.length() + " bytes)");
                    }
                } else {
                    LOGGER.warning("Translation directory is empty or not accessible");
                }
                this.loadTranslationFiles(translationDir);
                this.addProgrammaticTranslations();
                LOGGER.info(String.format("Successfully loaded %d locales with %d total translation keys", this.loadedLocales.size(), this.getTotalKeyCount()));
                LOGGER.info("Loaded locales: " + String.join((CharSequence)", ", this.loadedLocales));
            }
            catch (Exception e) {
                LOGGER.log(Level.SEVERE, "Failed to load translations", e);
                throw new RuntimeException("Translation loading failed", e);
            }
        });
    }

    @NotNull
    public Optional<List<String>> getRawTranslation(@NotNull String key, @NotNull String locale) {
        Map<String, List<String>> localeMap = this.translations.get(key);
        if (localeMap == null) {
            return Optional.empty();
        }
        List<String> translation = localeMap.get(locale);
        if (translation != null && !translation.isEmpty()) {
            return Optional.of(new ArrayList<String>(translation));
        }
        if (!locale.equals(this.configuration.defaultLocale()) && (translation = localeMap.get(this.configuration.defaultLocale())) != null && !translation.isEmpty()) {
            return Optional.of(new ArrayList<String>(translation));
        }
        return Optional.empty();
    }

    public boolean hasKey(@NotNull String key, @NotNull String locale) {
        return this.getRawTranslation(key, locale).isPresent();
    }

    @NotNull
    public Set<String> getAllKeys() {
        return new HashSet<String>(this.translations.keySet());
    }

    @NotNull
    public Set<String> getLoadedLocales() {
        return new HashSet<String>(this.loadedLocales);
    }

    public int getTotalKeyCount() {
        return this.translations.size();
    }

    @NotNull
    public Set<String> getMissingKeys(@NotNull String locale) {
        HashSet<String> missingKeys = new HashSet<String>();
        String defaultLocale = this.configuration.defaultLocale();
        for (Map.Entry<String, Map<String, List<String>>> entry : this.translations.entrySet()) {
            boolean missingInDefault;
            String key = entry.getKey();
            Map<String, List<String>> localeMap = entry.getValue();
            List<String> localeTranslation = localeMap.get(locale);
            List<String> defaultTranslation = localeMap.get(defaultLocale);
            boolean missingInLocale = localeTranslation == null || localeTranslation.isEmpty();
            boolean bl = missingInDefault = defaultTranslation == null || defaultTranslation.isEmpty();
            if (missingInLocale && missingInDefault) {
                missingKeys.add(key);
                continue;
            }
            if (!missingInLocale || locale.equals(defaultLocale)) continue;
            missingKeys.add(key);
        }
        return missingKeys;
    }

    @NotNull
    public Map<String, Map<String, List<String>>> getAllTranslations() {
        HashMap<String, Map<String, List<String>>> copy = new HashMap<String, Map<String, List<String>>>();
        for (Map.Entry<String, Map<String, List<String>>> entry : this.translations.entrySet()) {
            copy.put(entry.getKey(), new HashMap<String, List<String>>(entry.getValue()));
        }
        return copy;
    }

    @NotNull
    private File getTranslationDirectory() {
        return new File(this.plugin.getDataFolder(), this.configuration.translationDirectory());
    }

    private void ensureDirectoryExists(@NotNull File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new RuntimeException("Failed to create translation directory: " + directory.getAbsolutePath());
        }
    }

    private void extractResourceFiles(@NotNull File translationDir) {
        String translationPath = this.configuration.translationDirectory();
        this.extractUsingBukkitSaveResource(translationDir, translationPath);
        this.extractUsingJarScanning(translationDir, translationPath);
    }

    private void extractUsingBukkitSaveResource(@NotNull File translationDir, @NotNull String translationPath) {
        String[] commonLocales = new String[]{"en_US", "en_GB", "de_DE", "es_ES", "fr_FR", "ja_JP", "ko_KR", "zh_CN", "zh_TW", "pt_BR", "ru_RU", "it_IT", "nl_NL", "pl_PL", "tr_TR", "sv_SE", "no_NO", "da_DK", "en", "de"};
        Set<String> supportedLocales = this.configuration.supportedLocales();
        boolean filterEnabled = !supportedLocales.isEmpty();
        int extractedCount = 0;
        int skippedCount = 0;
        for (String locale : commonLocales) {
            if (filterEnabled && !supportedLocales.contains(locale)) {
                if (this.configuration.debugMode()) {
                    LOGGER.fine("Skipping extraction of unsupported locale: " + locale);
                }
                ++skippedCount;
                continue;
            }
            String resourcePath = translationPath + "/" + locale + ".yml";
            File targetFile = new File(translationDir, locale + ".yml");
            if (targetFile.exists()) continue;
            try {
                InputStream resourceStream = this.plugin.getResource(resourcePath);
                if (resourceStream == null) continue;
                Files.copy(resourceStream, targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                resourceStream.close();
                LOGGER.info("Extracted translation file via Bukkit: " + locale + ".yml");
                ++extractedCount;
            }
            catch (Exception e) {
                LOGGER.fine("Could not extract " + locale + ".yml via Bukkit: " + e.getMessage());
            }
        }
        if (extractedCount > 0) {
            LOGGER.info("Extracted " + extractedCount + " translation files via Bukkit saveResource");
        }
        if (skippedCount > 0 && this.configuration.debugMode()) {
            LOGGER.info("Skipped " + skippedCount + " unsupported locales during extraction");
        }
    }

    private void extractUsingJarScanning(@NotNull File translationDir, @NotNull String translationPath) {
        try {
            File jarFile = this.findPluginJarFile();
            if (jarFile == null) {
                LOGGER.fine("Could not locate plugin JAR file for scanning");
                return;
            }
            LOGGER.info("Scanning JAR for additional translation files: " + jarFile.getName());
            Set<String> supportedLocales = this.configuration.supportedLocales();
            boolean filterEnabled = !supportedLocales.isEmpty();
            try (JarFile jar = new JarFile(jarFile);){
                Enumeration<JarEntry> entries = jar.entries();
                int extractedCount = 0;
                int skippedCount = 0;
                while (entries.hasMoreElements()) {
                    JarEntry entry = entries.nextElement();
                    String entryName = entry.getName();
                    if (!entryName.startsWith(translationPath + "/") || !this.isTranslationFile(entryName) || entry.isDirectory()) continue;
                    String fileName = entryName.substring(entryName.lastIndexOf(47) + 1);
                    String locale = this.extractLocaleFromFilename(fileName);
                    if (filterEnabled && !supportedLocales.contains(locale)) {
                        if (this.configuration.debugMode()) {
                            LOGGER.fine("Skipping JAR extraction of unsupported locale: " + locale);
                        }
                        ++skippedCount;
                        continue;
                    }
                    File targetFile = new File(translationDir, fileName);
                    if (targetFile.exists()) continue;
                    InputStream in = jar.getInputStream(entry);
                    try {
                        if (in == null) continue;
                        Files.copy(in, targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                        LOGGER.info("Extracted translation file via JAR scan: " + fileName);
                        ++extractedCount;
                    }
                    finally {
                        if (in == null) continue;
                        in.close();
                    }
                }
                if (extractedCount > 0) {
                    LOGGER.info("Extracted " + extractedCount + " additional translation files from JAR");
                }
                if (skippedCount > 0 && this.configuration.debugMode()) {
                    LOGGER.info("Skipped " + skippedCount + " unsupported locales during JAR extraction");
                }
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to scan JAR for translation files: " + e.getMessage(), e);
        }
    }

    @Nullable
    private File findPluginJarFile() {
        File jarFile;
        try {
            CodeSource codeSource = this.plugin.getClass().getProtectionDomain().getCodeSource();
            if (codeSource != null && codeSource.getLocation() != null && (jarFile = new File(codeSource.getLocation().toURI())).exists() && jarFile.getName().endsWith(".jar")) {
                return jarFile;
            }
        }
        catch (Exception e) {
            LOGGER.fine("Could not get JAR via code source: " + e.getMessage());
        }
        try {
            Method getFileMethod = this.plugin.getClass().getMethod("getFile", new Class[0]);
            getFileMethod.setAccessible(true);
            jarFile = (File)getFileMethod.invoke((Object)this.plugin, new Object[0]);
            if (jarFile != null && jarFile.exists()) {
                return jarFile;
            }
        }
        catch (Exception e) {
            LOGGER.fine("Could not get JAR via getFile(): " + e.getMessage());
        }
        try {
            File pluginsFolder = this.plugin.getDataFolder().getParentFile();
            String pluginName = this.plugin.getName();
            File[] jarFiles = pluginsFolder.listFiles((dir, name) -> name.toLowerCase().contains(pluginName.toLowerCase()) && name.endsWith(".jar"));
            if (jarFiles != null && jarFiles.length > 0) {
                return jarFiles[0];
            }
        }
        catch (Exception e) {
            LOGGER.fine("Could not find JAR in plugins folder: " + e.getMessage());
        }
        return null;
    }

    private boolean isTranslationFile(@NotNull String fileName) {
        String lowerName = fileName.toLowerCase();
        return lowerName.endsWith(".yml") || lowerName.endsWith(".yaml") || lowerName.endsWith(".json");
    }

    @NotNull
    private String extractLocaleFromFilename(@NotNull String fileName) {
        int lastDot = fileName.lastIndexOf(46);
        return lastDot > 0 ? fileName.substring(0, lastDot) : fileName;
    }

    private void loadTranslationFiles(@NotNull File translationDir) {
        Stream<Path> paths;
        this.yamlLocales.clear();
        try {
            paths = Files.walk(translationDir.toPath(), new FileVisitOption[0]);
            try {
                paths.filter(path -> {
                    String name = path.toString().toLowerCase();
                    return name.endsWith(".yml") || name.endsWith(".yaml");
                }).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(this::loadTranslationFile);
            }
            finally {
                if (paths != null) {
                    paths.close();
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to scan translation directory for YAML files", e);
        }
        try {
            paths = Files.walk(translationDir.toPath(), new FileVisitOption[0]);
            try {
                paths.filter(path -> path.toString().toLowerCase().endsWith(".json")).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(this::loadTranslationFile);
            }
            finally {
                if (paths != null) {
                    paths.close();
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to scan translation directory for JSON files", e);
        }
    }

    private void loadTranslationFile(@NotNull Path filePath) {
        FileType fileType = this.detectFileType(filePath);
        switch (fileType.ordinal()) {
            case 1: {
                this.loadYamlFile(filePath);
                break;
            }
            case 0: {
                this.loadJsonFile(filePath);
                break;
            }
            default: {
                if (!this.configuration.debugMode()) break;
                LOGGER.warning("Unsupported file type: " + String.valueOf(filePath.getFileName()));
            }
        }
    }

    private void loadYamlFile(@NotNull Path filePath) {
        String fileName = filePath.getFileName().toString();
        String locale = fileName.substring(0, fileName.lastIndexOf(46));
        if (!this.configuration.supportedLocales().isEmpty() && !this.configuration.supportedLocales().contains(locale)) {
            if (this.configuration.debugMode()) {
                LOGGER.info("Skipping unsupported locale: " + locale);
            }
            return;
        }
        try {
            Map data;
            LOGGER.info("Loading YAML translation file: " + fileName);
            try (InputStream inputStream = Files.newInputStream(filePath, new OpenOption[0]);){
                data = (Map)this.yaml.load(inputStream);
            }
            if (data == null) {
                LOGGER.warning("Translation file is empty: " + fileName);
                return;
            }
            Map<String, Object> flattened = this.flattenMap(data);
            for (Map.Entry<String, Object> entry : flattened.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                List<String> messages = this.convertToStringList(value);
                if (messages.isEmpty()) continue;
                this.translations.computeIfAbsent(key, k -> new ConcurrentHashMap()).put(locale, messages);
            }
            this.loadedLocales.add(locale);
            this.yamlLocales.add(locale);
            if (this.configuration.debugMode()) {
                LOGGER.info(String.format("Loaded %d keys for locale %s from YAML", flattened.size(), locale));
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Failed to load YAML translation file: " + fileName, e);
        }
    }

    private void loadJsonFile(@NotNull Path filePath) {
        String fileName = filePath.getFileName().toString();
        String locale = fileName.substring(0, fileName.lastIndexOf(46));
        if (!this.configuration.supportedLocales().isEmpty() && !this.configuration.supportedLocales().contains(locale)) {
            if (this.configuration.debugMode()) {
                LOGGER.info("Skipping unsupported locale: " + locale);
            }
            return;
        }
        if (this.yamlLocales.contains(locale)) {
            if (this.configuration.debugMode()) {
                LOGGER.info("Skipping JSON file for locale " + locale + " (YAML takes precedence)");
            }
            return;
        }
        try {
            Map data;
            LOGGER.info("Loading JSON translation file: " + fileName);
            try (InputStream inputStream = Files.newInputStream(filePath, new OpenOption[0]);){
                data = (Map)this.jsonMapper.readValue(inputStream, (TypeReference)new TypeReference<Map<String, Object>>(this){});
            }
            if (data == null || data.isEmpty()) {
                LOGGER.warning("Translation file is empty: " + fileName);
                return;
            }
            Map<String, Object> flattened = this.flattenMap(data);
            for (Map.Entry<String, Object> entry : flattened.entrySet()) {
                String key = entry.getKey();
                Object value = entry.getValue();
                List<String> messages = this.convertToStringList(value);
                if (messages.isEmpty()) continue;
                this.translations.computeIfAbsent(key, k -> new ConcurrentHashMap()).put(locale, messages);
            }
            this.loadedLocales.add(locale);
            if (this.configuration.debugMode()) {
                LOGGER.info(String.format("Loaded %d keys for locale %s from JSON", flattened.size(), locale));
            }
        }
        catch (Exception e) {
            LOGGER.log(Level.WARNING, "Failed to load JSON translation file: " + fileName, e);
        }
    }

    @NotNull
    private Map<String, Object> flattenMap(@NotNull Map<String, Object> map) {
        HashMap<String, Object> flattened = new HashMap<String, Object>();
        this.flattenMapRecursive("", map, flattened);
        return flattened;
    }

    private void flattenMapRecursive(@NotNull String prefix, @NotNull Map<String, Object> map, @NotNull Map<String, Object> result) {
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String key = prefix.isEmpty() ? entry.getKey() : prefix + "." + entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Map) {
                this.flattenMapRecursive(key, (Map)value, result);
                continue;
            }
            result.put(key, value);
        }
    }

    @NotNull
    private List<String> convertToStringList(Object value) {
        if (value == null) {
            return Collections.emptyList();
        }
        if (value instanceof List) {
            List list = (List)value;
            return list.stream().map(Object::toString).toList();
        }
        return List.of(value.toString());
    }

    private void addProgrammaticTranslations() {
        this.addInternalTranslation("r18n.reload.success", "en_US", "<green>\u2713 Translations reloaded successfully!</green>");
        this.addInternalTranslation("r18n.reload.failure", "en_US", "<red>\u2717 Failed to reload translations!</red>");
        this.addInternalTranslation("r18n.missing.none", "en_US", "<green>\u2713 No missing translation keys found!</green>");
        this.addInternalTranslation("r18n.missing.header", "en_US", "<yellow>Found {count} missing keys for locale '{locale}':</yellow>");
        this.addInternalTranslation("r18n.missing.key", "en_US", "<red> \u2717 {key}</red>");
        this.addInternalTranslation("r18n.key.missing", "en_US", "<gold>Missing key: <red>{key}</red></gold>");
        this.addInternalTranslation("r18n.reload.success", "de_DE", "<green>\u2713 \u00dcbersetzungen erfolgreich neu geladen!</green>");
        this.addInternalTranslation("r18n.reload.failure", "de_DE", "<red>\u2717 Fehler beim Neuladen der \u00dcbersetzungen!</red>");
        this.addInternalTranslation("r18n.missing.none", "de_DE", "<green>\u2713 Keine fehlenden \u00dcbersetzungsschl\u00fcssel gefunden!</green>");
        this.addInternalTranslation("r18n.missing.header", "de_DE", "<yellow>{count} fehlende Schl\u00fcssel f\u00fcr Sprache '{locale}' gefunden:</yellow>");
        this.addInternalTranslation("r18n.key.missing", "de_DE", "<gold>Fehlender Schl\u00fcssel: <red>{key}</red></gold>");
        this.addInternalTranslation("r18n.reload.success", "es_ES", "<green>\u2713 \u00a1Traducciones recargadas exitosamente!</green>");
        this.addInternalTranslation("r18n.reload.failure", "es_ES", "<red>\u2717 \u00a1Error al recargar las traducciones!</red>");
        this.addInternalTranslation("r18n.missing.none", "es_ES", "<green>\u2713 \u00a1No se encontraron claves de traducci\u00f3n faltantes!</green>");
        this.addInternalTranslation("r18n.missing.header", "es_ES", "<yellow>Se encontraron {count} claves faltantes para el idioma '{locale}':</yellow>");
        this.addInternalTranslation("r18n.key.missing", "es_ES", "<gold>Clave faltante: <red>{key}</red></gold>");
    }

    private void addInternalTranslation(@NotNull String key, @NotNull String locale, @NotNull String message) {
        this.translations.computeIfAbsent(key, k -> new ConcurrentHashMap()).put(locale, List.of(message));
    }

    public static enum FileType {
        JSON,
        YAML,
        UNKNOWN;

    }
}

