/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.jextranslate.core;

import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jetbrains.annotations.NotNull;

public final class TranslationFileWatcher
implements Runnable {
    private static final Logger LOGGER = Logger.getLogger(TranslationFileWatcher.class.getName());
    private static final long DEFAULT_DEBOUNCE_MS = 500L;
    private final WatchService watchService;
    private final Path translationsDir;
    private final Runnable onReload;
    private final long debounceMs;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final AtomicLong lastReloadTime = new AtomicLong(0L);

    public TranslationFileWatcher(@NotNull Path translationsDir, @NotNull Runnable onReload) throws IOException {
        this(translationsDir, onReload, 500L);
    }

    public TranslationFileWatcher(@NotNull Path translationsDir, @NotNull Runnable onReload, long debounceMs) throws IOException {
        this.translationsDir = translationsDir;
        this.onReload = onReload;
        this.debounceMs = debounceMs;
        this.watchService = FileSystems.getDefault().newWatchService();
        translationsDir.register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
        LOGGER.info("File watcher registered for directory: " + String.valueOf(translationsDir));
    }

    @Override
    public void run() {
        LOGGER.info("Translation file watcher started");
        while (this.running.get()) {
            try {
                WatchKey key = this.watchService.poll(1L, TimeUnit.SECONDS);
                if (key == null) continue;
                boolean shouldReload = false;
                for (WatchEvent<?> event : key.pollEvents()) {
                    WatchEvent.Kind<?> kind = event.kind();
                    if (kind == StandardWatchEventKinds.OVERFLOW) {
                        LOGGER.warning("File watcher overflow event - some events may have been lost");
                        shouldReload = true;
                        continue;
                    }
                    WatchEvent<?> pathEvent = event;
                    Path fileName = (Path)pathEvent.context();
                    if (!this.isTranslationFile(fileName.toString())) continue;
                    LOGGER.info(String.format("Translation file %s: %s", this.getEventTypeName(kind), fileName));
                    shouldReload = true;
                }
                boolean valid = key.reset();
                if (!valid) {
                    LOGGER.warning("Watch key is no longer valid - directory may have been deleted");
                    break;
                }
                if (!shouldReload) continue;
                this.triggerReloadWithDebounce();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.info("File watcher interrupted");
                break;
            }
            catch (ClosedWatchServiceException e) {
                LOGGER.info("Watch service closed");
                break;
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "Error in file watcher loop", e);
            }
        }
        LOGGER.info("Translation file watcher stopped");
    }

    public void stop() {
        this.running.set(false);
        try {
            this.watchService.close();
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Error closing watch service", e);
        }
    }

    public boolean isRunning() {
        return this.running.get();
    }

    private void triggerReloadWithDebounce() {
        long lastTime;
        long currentTime = System.currentTimeMillis();
        if (currentTime - (lastTime = this.lastReloadTime.get()) >= this.debounceMs) {
            if (this.lastReloadTime.compareAndSet(lastTime, currentTime)) {
                LOGGER.info("Triggering translation reload due to file changes");
                try {
                    this.onReload.run();
                }
                catch (Exception e) {
                    LOGGER.log(Level.SEVERE, "Error during translation reload callback", e);
                }
            }
        } else {
            LOGGER.fine("Debouncing reload - too soon since last reload");
        }
    }

    private boolean isTranslationFile(@NotNull String fileName) {
        String lowerName = fileName.toLowerCase();
        return lowerName.endsWith(".yml") || lowerName.endsWith(".yaml") || lowerName.endsWith(".json");
    }

    @NotNull
    private String getEventTypeName(@NotNull WatchEvent.Kind<?> kind) {
        if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
            return "created";
        }
        if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
            return "modified";
        }
        if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
            return "deleted";
        }
        return "changed";
    }
}

