/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.hibernate.util;

import de.jexcellence.hibernate.type.DatabaseType;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Locale;
import java.util.Properties;
import java.util.stream.Collectors;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ConfigLoader {
    private static final String FALLBACK_PATH = "hibernate.properties";
    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigLoader.class);

    @NotNull
    public Properties loadAndValidateProperties(@NotNull String filePath) throws IOException {
        Properties rawProperties = new Properties();
        this.loadProperties(filePath, rawProperties);
        return this.buildFinalProperties(rawProperties);
    }

    private void loadProperties(@NotNull String filePath, @NotNull Properties target) throws IOException {
        Path path = Path.of(filePath, new String[0]);
        try (InputStream stream = Files.newInputStream(path, new OpenOption[0]);){
            target.load(stream);
            LOGGER.info("Loaded Hibernate properties from {}", (Object)path.toAbsolutePath());
            return;
        }
        catch (IOException e) {
            LOGGER.warn("Failed to read properties from {}, attempting bundled fallback", (Object)path.toAbsolutePath());
            try (InputStream stream2 = this.getClass().getClassLoader().getResourceAsStream(FALLBACK_PATH);){
                if (stream2 == null) {
                    throw new IOException("Fallback properties file '%s' is not available".formatted(FALLBACK_PATH));
                }
                target.load(stream2);
                LOGGER.info("Loaded Hibernate properties from bundled resource {}", (Object)FALLBACK_PATH);
            }
            return;
        }
    }

    @NotNull
    private Properties buildFinalProperties(@NotNull Properties rawProperties) {
        Properties finalProperties = new Properties();
        DatabaseType databaseType = this.resolveDatabaseType(rawProperties.getProperty("database.type"));
        String prefix = this.getDatabasePrefix(databaseType);
        LOGGER.info("Using database type: {}", (Object)databaseType);
        this.applyDatabaseProperties(rawProperties, finalProperties, prefix, databaseType);
        this.applyHibernateSettings(rawProperties, finalProperties);
        return finalProperties;
    }

    private void applyDatabaseProperties(@NotNull Properties source, @NotNull Properties target, @NotNull String prefix, @NotNull DatabaseType databaseType) {
        String url = source.getProperty(prefix + ".url");
        if (url != null && !url.isBlank()) {
            target.setProperty("jakarta.persistence.jdbc.url", url);
        } else if (databaseType == DatabaseType.H2) {
            target.setProperty("jakarta.persistence.jdbc.url", "jdbc:h2:mem:testdb;DB_CLOSE_DELAY=-1");
        }
        String driver = source.getProperty(prefix + ".driver");
        target.setProperty("jakarta.persistence.jdbc.driver", driver != null && !driver.isBlank() ? driver : this.getDefaultDriver(databaseType));
        String dialect = source.getProperty(prefix + ".dialect");
        target.setProperty("hibernate.dialect", dialect != null && !dialect.isBlank() ? dialect : this.getDefaultDialect(databaseType));
        this.applyCredentials(source, target, prefix, databaseType);
    }

    private void applyCredentials(@NotNull Properties source, @NotNull Properties target, @NotNull String prefix, @NotNull DatabaseType databaseType) {
        String username = source.getProperty(prefix + ".username");
        String password = source.getProperty(prefix + ".password");
        if (databaseType == DatabaseType.H2) {
            target.setProperty("jakarta.persistence.jdbc.user", username != null ? username : "sa");
            target.setProperty("jakarta.persistence.jdbc.password", password != null ? password : "");
        } else {
            if (username == null || username.isBlank()) {
                throw new IllegalArgumentException("Database username is required for %s. Set '%s.username' in configuration.".formatted(new Object[]{databaseType, prefix}));
            }
            if (password == null) {
                throw new IllegalArgumentException("Database password is required for %s. Set '%s.password' in configuration.".formatted(new Object[]{databaseType, prefix}));
            }
            target.setProperty("jakarta.persistence.jdbc.user", username);
            target.setProperty("jakarta.persistence.jdbc.password", password);
        }
    }

    private void applyHibernateSettings(@NotNull Properties source, @NotNull Properties target) {
        String hbm2ddl = source.getProperty("hibernate.hbm2ddl.auto");
        if (hbm2ddl != null) {
            target.setProperty("hibernate.hbm2ddl.auto", hbm2ddl);
        }
        this.copyIfPresent(source, target, "hibernate.show_sql", "hibernate.show_sql");
        this.copyIfPresent(source, target, "hibernate.format_sql", "hibernate.format_sql");
        this.copyIfPresent(source, target, "hibernate.use_sql_comments", "hibernate.use_sql_comments");
        this.copyIfPresent(source, target, "hibernate.highlight_sql", "hibernate.highlight_sql");
        this.copyIfPresent(source, target, "hibernate.jdbc.batch_size", "hibernate.jdbc.batch_size");
        this.copyIfPresent(source, target, "hibernate.order_inserts", "hibernate.order_inserts");
        this.copyIfPresent(source, target, "hibernate.order_updates", "hibernate.order_updates");
        this.copyIfPresent(source, target, "hibernate.generate_statistics", "hibernate.generate_statistics");
    }

    private void copyIfPresent(@NotNull Properties source, @NotNull Properties target, @NotNull String sourceKey, @NotNull String targetKey) {
        String value = source.getProperty(sourceKey);
        if (value != null) {
            target.setProperty(targetKey, value);
        }
    }

    @NotNull
    private String getDatabasePrefix(@NotNull DatabaseType databaseType) {
        return switch (databaseType) {
            default -> throw new MatchException(null, null);
            case DatabaseType.H2 -> "h2";
            case DatabaseType.MYSQL -> "mysql";
            case DatabaseType.MARIADB -> "mariadb";
            case DatabaseType.POSTGRESQL -> "postgresql";
            case DatabaseType.ORACLE -> "oracle";
            case DatabaseType.MSSQL_SERVER -> "mssql";
            case DatabaseType.SQLITE -> "sqlite";
            case DatabaseType.HSQLDB -> "hsqldb";
        };
    }

    @NotNull
    private String getDefaultDriver(@NotNull DatabaseType databaseType) {
        return switch (databaseType) {
            default -> throw new MatchException(null, null);
            case DatabaseType.H2 -> "org.h2.Driver";
            case DatabaseType.MYSQL -> "com.mysql.cj.jdbc.Driver";
            case DatabaseType.MARIADB -> "org.mariadb.jdbc.Driver";
            case DatabaseType.POSTGRESQL -> "org.postgresql.Driver";
            case DatabaseType.ORACLE -> "oracle.jdbc.OracleDriver";
            case DatabaseType.MSSQL_SERVER -> "com.microsoft.sqlserver.jdbc.SQLServerDriver";
            case DatabaseType.SQLITE -> "org.sqlite.JDBC";
            case DatabaseType.HSQLDB -> "org.hsqldb.jdbc.JDBCDriver";
        };
    }

    @NotNull
    private String getDefaultDialect(@NotNull DatabaseType databaseType) {
        return switch (databaseType) {
            default -> throw new MatchException(null, null);
            case DatabaseType.H2 -> "org.hibernate.dialect.H2Dialect";
            case DatabaseType.MYSQL -> "org.hibernate.dialect.MySQLDialect";
            case DatabaseType.MARIADB -> "org.hibernate.dialect.MariaDBDialect";
            case DatabaseType.POSTGRESQL -> "org.hibernate.dialect.PostgreSQLDialect";
            case DatabaseType.ORACLE -> "org.hibernate.dialect.OracleDialect";
            case DatabaseType.MSSQL_SERVER -> "org.hibernate.dialect.SQLServerDialect";
            case DatabaseType.SQLITE -> "org.hibernate.community.dialect.SQLiteDialect";
            case DatabaseType.HSQLDB -> "org.hibernate.dialect.HSQLDialect";
        };
    }

    @NotNull
    private DatabaseType resolveDatabaseType(@Nullable String type) {
        if (type == null || type.isBlank()) {
            LOGGER.info("No database.type specified, defaulting to H2");
            return DatabaseType.H2;
        }
        String normalizedType = type.toUpperCase(Locale.ROOT).replace("-", "_");
        try {
            return DatabaseType.valueOf(normalizedType);
        }
        catch (IllegalArgumentException e) {
            String supportedTypes = Arrays.stream(DatabaseType.values()).map(Enum::name).collect(Collectors.joining(", "));
            throw new IllegalArgumentException("Unsupported database type '%s'. Supported types: %s".formatted(type, supportedTypes));
        }
    }
}

