/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.hibernate.repository;

import de.jexcellence.hibernate.repository.BaseRepository;
import de.jexcellence.remapped.com.github.benmanes.caffeine.cache.Cache;
import de.jexcellence.remapped.com.github.benmanes.caffeine.cache.Caffeine;
import de.jexcellence.remapped.com.github.benmanes.caffeine.cache.stats.CacheStats;
import jakarta.persistence.EntityManagerFactory;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import org.jetbrains.annotations.NotNull;

public class CachedRepository<T, ID, K>
extends BaseRepository<T, ID> {
    private final ExecutorService executor;
    private final Cache<K, T> keyCache;
    private final Cache<ID, T> idCache;
    private final Function<T, K> keyExtractor;
    private final Function<T, ID> idExtractor;

    public CachedRepository(@NotNull ExecutorService executor, @NotNull EntityManagerFactory entityManagerFactory, @NotNull Class<T> entityClass, @NotNull Function<T, K> keyExtractor) {
        this(executor, entityManagerFactory, entityClass, keyExtractor, Duration.ofMinutes(30L), 10000);
    }

    public CachedRepository(@NotNull ExecutorService executor, @NotNull EntityManagerFactory entityManagerFactory, @NotNull Class<T> entityClass, @NotNull Function<T, K> keyExtractor, @NotNull Duration expiration, int maxSize) {
        super(executor, entityManagerFactory, entityClass);
        this.executor = executor;
        this.keyExtractor = keyExtractor;
        this.idExtractor = this.createIdExtractor();
        this.keyCache = Caffeine.newBuilder().expireAfterWrite(expiration).maximumSize((long)maxSize).recordStats().build();
        this.idCache = Caffeine.newBuilder().expireAfterWrite(expiration).maximumSize((long)maxSize).recordStats().build();
    }

    @Override
    @NotNull
    public Optional<T> findById(@NotNull ID id) {
        Object cached = this.idCache.getIfPresent(id);
        if (cached != null) {
            return Optional.of(cached);
        }
        Optional<Object> found = super.findById(id);
        found.ifPresent(entity -> {
            this.idCache.put(id, entity);
            this.keyCache.put(this.keyExtractor.apply(entity), entity);
        });
        return found;
    }

    @NotNull
    public Optional<T> findByKey(@NotNull K key) {
        Object cached = this.keyCache.getIfPresent(key);
        return Optional.ofNullable(cached);
    }

    @NotNull
    public Optional<T> findByKey(@NotNull String queryAttribute, @NotNull K key) {
        Object cached = this.keyCache.getIfPresent(key);
        if (cached != null) {
            return Optional.of(cached);
        }
        Optional<Object> found = super.findByAttribute(queryAttribute, key);
        found.ifPresent(entity -> {
            this.keyCache.put(key, entity);
            ID id = this.idExtractor.apply(entity);
            if (id != null) {
                this.idCache.put(id, entity);
            }
        });
        return found;
    }

    @NotNull
    public CompletableFuture<Optional<T>> findByKeyAsync(@NotNull String queryAttribute, @NotNull K key) {
        return CompletableFuture.supplyAsync(() -> this.findByKey(queryAttribute, key), this.executor);
    }

    @NotNull
    public T getOrCreate(@NotNull String queryAttribute, @NotNull K key, @NotNull Function<K, T> creator) {
        return (T)this.findByKey(queryAttribute, key).orElseGet(() -> this.create(creator.apply(key)));
    }

    @NotNull
    public CompletableFuture<T> getOrCreateAsync(@NotNull String queryAttribute, @NotNull K key, @NotNull Function<K, T> creator) {
        return CompletableFuture.supplyAsync(() -> this.getOrCreate(queryAttribute, key, creator), this.executor);
    }

    @Override
    @NotNull
    public List<T> findAll(int pageNumber, int pageSize) {
        ConcurrentMap cachedEntities = this.keyCache.asMap();
        if (!cachedEntities.isEmpty() && cachedEntities.size() >= pageSize) {
            int size = Math.max(pageSize, 1);
            int offset = Math.max(pageNumber, 0) * size;
            return cachedEntities.values().stream().skip(offset).limit(size).toList();
        }
        List<Object> entities = super.findAll(pageNumber, pageSize);
        entities.forEach(this::cacheEntity);
        return entities;
    }

    @Override
    @NotNull
    public T create(@NotNull T entity) {
        T created = super.create(entity);
        this.cacheEntity(created);
        return created;
    }

    @Override
    @NotNull
    public List<T> createAll(@NotNull Collection<T> entities) {
        List<Object> created = super.createAll(entities);
        created.forEach(this::cacheEntity);
        return created;
    }

    @Override
    @NotNull
    public T update(@NotNull T entity) {
        T updated = super.update(entity);
        this.cacheEntity(updated);
        return updated;
    }

    @Override
    @NotNull
    public T save(@NotNull T entity) {
        T saved = super.save(entity);
        this.cacheEntity(saved);
        return saved;
    }

    @Override
    public boolean delete(@NotNull ID id) {
        boolean deleted = super.delete(id);
        if (deleted) {
            this.evictById(id);
        }
        return deleted;
    }

    @Override
    public void deleteEntity(@NotNull T entity) {
        super.deleteEntity(entity);
        this.evict(entity);
    }

    public void evict(@NotNull T entity) {
        this.keyCache.invalidate(this.keyExtractor.apply(entity));
        ID id = this.idExtractor.apply(entity);
        if (id != null) {
            this.idCache.invalidate(id);
        }
    }

    public void evictById(@NotNull ID id) {
        Object cached = this.idCache.getIfPresent(id);
        if (cached != null) {
            this.keyCache.invalidate(this.keyExtractor.apply(cached));
        }
        this.idCache.invalidate(id);
    }

    public void evictByKey(@NotNull K key) {
        ID id;
        Object cached = this.keyCache.getIfPresent(key);
        if (cached != null && (id = this.idExtractor.apply(cached)) != null) {
            this.idCache.invalidate(id);
        }
        this.keyCache.invalidate(key);
    }

    public void evictAll() {
        this.keyCache.invalidateAll();
        this.idCache.invalidateAll();
    }

    public void preload() {
        super.findAll().forEach(this::cacheEntity);
    }

    @NotNull
    public CompletableFuture<Void> preloadAsync() {
        return CompletableFuture.runAsync(this::preload, this.executor);
    }

    @NotNull
    public Map<K, T> getCachedByKey() {
        return Map.copyOf(this.keyCache.asMap());
    }

    @NotNull
    public Map<ID, T> getCachedById() {
        return Map.copyOf(this.idCache.asMap());
    }

    public long getCacheSize() {
        return this.keyCache.estimatedSize();
    }

    @NotNull
    public CacheStats getKeyCacheStats() {
        return this.keyCache.stats();
    }

    @NotNull
    public CacheStats getIdCacheStats() {
        return this.idCache.stats();
    }

    private void cacheEntity(@NotNull T entity) {
        this.keyCache.put(this.keyExtractor.apply(entity), entity);
        ID id = this.idExtractor.apply(entity);
        if (id != null) {
            this.idCache.put(id, entity);
        }
    }

    private Function<T, ID> createIdExtractor() {
        return entity -> {
            try {
                Method method = entity.getClass().getMethod("getId", new Class[0]);
                return method.invoke(entity, new Object[0]);
            }
            catch (ReflectiveOperationException e) {
                return null;
            }
        };
    }
}

