/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.hibernate.repository;

import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;
import jakarta.persistence.EntityTransaction;
import jakarta.persistence.TypedQuery;
import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.CriteriaDelete;
import jakarta.persistence.criteria.CriteriaQuery;
import jakarta.persistence.criteria.Expression;
import jakarta.persistence.criteria.Order;
import jakarta.persistence.criteria.Path;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;
import jakarta.persistence.criteria.Selection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class BaseRepository<T, ID> {
    private final Logger logger;
    private final Class<T> entityClass;
    private final EntityManagerFactory entityManagerFactory;
    private final ExecutorService executorService;

    public BaseRepository(@NotNull ExecutorService executor, @NotNull EntityManagerFactory entityManagerFactory, @NotNull Class<T> entityClass) {
        this.entityManagerFactory = entityManagerFactory;
        this.entityClass = entityClass;
        this.executorService = executor;
        this.logger = Logger.getLogger(entityClass.getName());
    }

    @NotNull
    public CompletableFuture<T> createAsync(@NotNull T entity) {
        return CompletableFuture.supplyAsync(() -> this.create(entity), this.executorService);
    }

    @NotNull
    public CompletableFuture<T> updateAsync(@NotNull T entity) {
        return CompletableFuture.supplyAsync(() -> this.update(entity), this.executorService);
    }

    @NotNull
    public CompletableFuture<Boolean> deleteAsync(@NotNull ID id) {
        return CompletableFuture.supplyAsync(() -> this.delete(id), this.executorService);
    }

    @NotNull
    public CompletableFuture<Optional<T>> findByIdAsync(@NotNull ID id) {
        return CompletableFuture.supplyAsync(() -> this.findById(id), this.executorService);
    }

    @NotNull
    public CompletableFuture<List<T>> findAllAsync(int pageNumber, int pageSize) {
        return CompletableFuture.supplyAsync(() -> this.findAll(pageNumber, pageSize), this.executorService);
    }

    @NotNull
    public T create(@NotNull T entity) {
        return (T)this.executeInTransaction(em -> {
            em.persist(entity);
            return entity;
        });
    }

    @NotNull
    public List<T> createAll(@NotNull Collection<T> entities) {
        return this.executeInTransaction(em -> {
            ArrayList result = new ArrayList(entities.size());
            int count = 0;
            for (Object entity : entities) {
                em.persist(entity);
                result.add(entity);
                if (++count % 50 != 0) continue;
                em.flush();
                em.clear();
            }
            return result;
        });
    }

    @NotNull
    public CompletableFuture<List<T>> createAllAsync(@NotNull Collection<T> entities) {
        return CompletableFuture.supplyAsync(() -> this.createAll(entities), this.executorService);
    }

    @NotNull
    public T update(@NotNull T entity) {
        return (T)this.executeInTransaction(em -> em.merge(entity));
    }

    @NotNull
    public List<T> updateAll(@NotNull Collection<T> entities) {
        return this.executeInTransaction(em -> {
            ArrayList<Object> result = new ArrayList<Object>(entities.size());
            int count = 0;
            for (Object entity : entities) {
                result.add(em.merge(entity));
                if (++count % 50 != 0) continue;
                em.flush();
                em.clear();
            }
            return result;
        });
    }

    @NotNull
    public T save(@NotNull T entity) {
        return (T)this.executeInTransaction(em -> {
            if (em.contains(entity)) {
                return em.merge(entity);
            }
            em.persist(entity);
            return entity;
        });
    }

    public boolean delete(@NotNull ID id) {
        return this.executeInTransaction(em -> {
            Object entity = em.find(this.entityClass, id);
            if (entity != null) {
                em.remove(entity);
                return true;
            }
            return false;
        });
    }

    public void deleteEntity(@NotNull T entity) {
        this.runInTransaction(em -> {
            Object managed = em.contains(entity) ? entity : em.merge(entity);
            em.remove(managed);
        });
    }

    public int deleteAll(@NotNull Collection<ID> ids) {
        if (ids.isEmpty()) {
            return 0;
        }
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaDelete delete = cb.createCriteriaDelete(this.entityClass);
            Root root = delete.from(this.entityClass);
            delete.where((Expression)root.get("id").in(ids));
            return em.createQuery(delete).executeUpdate();
        });
    }

    public int deleteByAttribute(@NotNull String attribute, @NotNull Object value) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaDelete delete = cb.createCriteriaDelete(this.entityClass);
            Root root = delete.from(this.entityClass);
            delete.where((Expression)cb.equal((Expression)root.get(attribute), value));
            return em.createQuery(delete).executeUpdate();
        });
    }

    @NotNull
    public Optional<T> findById(@NotNull ID id) {
        return this.executeInTransaction(em -> Optional.ofNullable(em.find(this.entityClass, id)));
    }

    public boolean existsById(@NotNull ID id) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(Long.class);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)cb.literal((Object)1L)).where((Expression)cb.equal((Expression)root.get("id"), id));
            return !em.createQuery(cq).setMaxResults(1).getResultList().isEmpty();
        });
    }

    public long count() {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(Long.class);
            cq.select((Selection)cb.count((Expression)cq.from(this.entityClass)));
            return (Long)em.createQuery(cq).getSingleResult();
        });
    }

    public long countByAttribute(@NotNull String attribute, @NotNull Object value) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(Long.class);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)cb.count((Expression)root)).where((Expression)cb.equal((Expression)root.get(attribute), value));
            return (Long)em.createQuery(cq).getSingleResult();
        });
    }

    @NotNull
    public List<T> findAll() {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            cq.select((Selection)cq.from(this.entityClass));
            return em.createQuery(cq).getResultList();
        });
    }

    @NotNull
    public List<T> findAll(int pageNumber, int pageSize) {
        return this.findAll(pageNumber, pageSize, null, true);
    }

    @NotNull
    public List<T> findAll(int pageNumber, int pageSize, @Nullable String sortBy, boolean ascending) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)root);
            if (sortBy != null && !sortBy.isBlank()) {
                cq.orderBy(new Order[]{ascending ? cb.asc((Expression)root.get(sortBy)) : cb.desc((Expression)root.get(sortBy))});
            }
            return em.createQuery(cq).setFirstResult(Math.max(pageNumber, 0) * Math.max(pageSize, 1)).setMaxResults(Math.max(pageSize, 1)).getResultList();
        });
    }

    @NotNull
    public List<T> findAllById(@NotNull Collection<ID> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyList();
        }
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)root).where((Expression)root.get("id").in(ids));
            return em.createQuery(cq).getResultList();
        });
    }

    @NotNull
    public Optional<T> findByAttribute(@NotNull String attribute, @NotNull Object value) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)root).where((Expression)cb.equal((Expression)root.get(attribute), value));
            return em.createQuery(cq).getResultStream().findFirst();
        });
    }

    @NotNull
    public List<T> findAllByAttribute(@NotNull String attribute, @NotNull Object value) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            cq.select((Selection)root).where((Expression)cb.equal((Expression)root.get(attribute), value));
            return em.createQuery(cq).getResultList();
        });
    }

    @NotNull
    public Optional<T> findByAttributes(@NotNull Map<String, Object> attributes) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            try {
                this.applyPredicates(attributes, cb, cq, root);
                return em.createQuery(cq).getResultStream().findFirst();
            }
            catch (RuntimeException e) {
                this.logger.log(Level.WARNING, "Failed to build attribute predicates", e);
                return Optional.empty();
            }
        });
    }

    @NotNull
    public List<T> findAllByAttributes(@NotNull Map<String, Object> attributes) {
        return this.executeInTransaction(em -> {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.entityClass);
            Root root = cq.from(this.entityClass);
            try {
                this.applyPredicates(attributes, cb, cq, root);
                return em.createQuery(cq).getResultList();
            }
            catch (RuntimeException e) {
                this.logger.log(Level.WARNING, "Failed to build attribute predicates", e);
                return Collections.emptyList();
            }
        });
    }

    @NotNull
    public CompletableFuture<List<T>> findAllByAttributesAsync(@NotNull Map<String, Object> attributes) {
        return CompletableFuture.supplyAsync(() -> this.findAllByAttributes(attributes), this.executorService);
    }

    @NotNull
    public QueryBuilder<T> query() {
        return new QueryBuilder(this);
    }

    public void refresh(@NotNull T entity) {
        this.runInTransaction(em -> em.refresh(em.contains(entity) ? entity : em.merge(entity)));
    }

    public void detach(@NotNull T entity) {
        this.runInTransaction(em -> {
            if (em.contains(entity)) {
                em.detach(entity);
            }
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NotNull
    public <R> R executeInTransaction(@NotNull Function<EntityManager, R> action) {
        EntityTransaction transaction = null;
        try (EntityManager em = this.entityManagerFactory.createEntityManager();){
            transaction = em.getTransaction();
            transaction.begin();
            R result = action.apply(em);
            transaction.commit();
            R r = result;
            return r;
        }
        catch (RuntimeException e) {
            this.rollbackQuietly(transaction);
            throw e;
        }
        catch (Exception e) {
            this.rollbackQuietly(transaction);
            throw new IllegalStateException("Repository operation failed", e);
        }
    }

    public void runInTransaction(@NotNull Consumer<EntityManager> action) {
        this.executeInTransaction(em -> {
            action.accept((EntityManager)em);
            return null;
        });
    }

    @NotNull
    protected Class<T> getEntityClass() {
        return this.entityClass;
    }

    @NotNull
    protected ExecutorService getExecutorService() {
        return this.executorService;
    }

    private void applyPredicates(@NotNull Map<String, Object> attributes, @NotNull CriteriaBuilder cb, @NotNull CriteriaQuery<T> cq, @NotNull Root<T> root) {
        ArrayList<Predicate> predicates = new ArrayList<Predicate>();
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            if (key.contains(".")) {
                String[] parts = key.split("\\.");
                Path path = root.get(parts[0]);
                for (int i = 1; i < parts.length; ++i) {
                    path = path.get(parts[i]);
                }
                predicates.add(cb.equal((Expression)path, value));
                continue;
            }
            predicates.add(cb.equal((Expression)root.get(key), value));
        }
        cq.select(root).where((Predicate[])predicates.toArray(Predicate[]::new));
    }

    private void rollbackQuietly(@Nullable EntityTransaction transaction) {
        if (transaction == null || !transaction.isActive()) {
            return;
        }
        try {
            transaction.rollback();
        }
        catch (RuntimeException e) {
            this.logger.log(Level.WARNING, "Failed to rollback transaction", e);
        }
    }

    public static class QueryBuilder<E> {
        private final BaseRepository<E, ?> repository;
        private final List<PredicateSpec> predicates = new ArrayList<PredicateSpec>();
        private String sortBy;
        private boolean ascending = true;
        private int offset = 0;
        private int limit = Integer.MAX_VALUE;

        QueryBuilder(BaseRepository<E, ?> repository) {
            this.repository = repository;
        }

        @NotNull
        public QueryBuilder<E> where(@NotNull String attribute, @NotNull Object value) {
            this.predicates.add(new PredicateSpec(attribute, value, PredicateType.EQUAL));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereLike(@NotNull String attribute, @NotNull String pattern) {
            this.predicates.add(new PredicateSpec(attribute, pattern, PredicateType.LIKE));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereIn(@NotNull String attribute, @NotNull Collection<?> values) {
            this.predicates.add(new PredicateSpec(attribute, values, PredicateType.IN));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereNotNull(@NotNull String attribute) {
            this.predicates.add(new PredicateSpec(attribute, null, PredicateType.NOT_NULL));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereNull(@NotNull String attribute) {
            this.predicates.add(new PredicateSpec(attribute, null, PredicateType.IS_NULL));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereGreaterThan(@NotNull String attribute, @NotNull Comparable<?> value) {
            this.predicates.add(new PredicateSpec(attribute, value, PredicateType.GREATER_THAN));
            return this;
        }

        @NotNull
        public QueryBuilder<E> whereLessThan(@NotNull String attribute, @NotNull Comparable<?> value) {
            this.predicates.add(new PredicateSpec(attribute, value, PredicateType.LESS_THAN));
            return this;
        }

        @NotNull
        public QueryBuilder<E> orderBy(@NotNull String attribute) {
            this.sortBy = attribute;
            this.ascending = true;
            return this;
        }

        @NotNull
        public QueryBuilder<E> orderByDesc(@NotNull String attribute) {
            this.sortBy = attribute;
            this.ascending = false;
            return this;
        }

        @NotNull
        public QueryBuilder<E> offset(int offset) {
            this.offset = Math.max(0, offset);
            return this;
        }

        @NotNull
        public QueryBuilder<E> limit(int limit) {
            this.limit = Math.max(1, limit);
            return this;
        }

        @NotNull
        public QueryBuilder<E> page(int pageNumber, int pageSize) {
            this.offset = Math.max(0, pageNumber) * Math.max(1, pageSize);
            this.limit = Math.max(1, pageSize);
            return this;
        }

        @NotNull
        public List<E> list() {
            return this.repository.executeInTransaction(this::executeQuery);
        }

        @NotNull
        public Optional<E> first() {
            this.limit = 1;
            return this.repository.executeInTransaction(em -> this.executeQuery((EntityManager)em).stream().findFirst());
        }

        public long count() {
            return this.repository.executeInTransaction(em -> {
                CriteriaBuilder cb = em.getCriteriaBuilder();
                CriteriaQuery cq = cb.createQuery(Long.class);
                Root root = cq.from(this.repository.getEntityClass());
                cq.select((Selection)cb.count((Expression)root));
                this.applyPredicates(cb, cq, root);
                return (Long)em.createQuery(cq).getSingleResult();
            });
        }

        public boolean exists() {
            return this.count() > 0L;
        }

        @NotNull
        public CompletableFuture<List<E>> listAsync() {
            return CompletableFuture.supplyAsync(this::list, this.repository.getExecutorService());
        }

        private List<E> executeQuery(EntityManager em) {
            CriteriaBuilder cb = em.getCriteriaBuilder();
            CriteriaQuery cq = cb.createQuery(this.repository.getEntityClass());
            Root root = cq.from(this.repository.getEntityClass());
            cq.select((Selection)root);
            this.applyPredicates(cb, cq, root);
            if (this.sortBy != null) {
                cq.orderBy(new Order[]{this.ascending ? cb.asc((Expression)root.get(this.sortBy)) : cb.desc((Expression)root.get(this.sortBy))});
            }
            TypedQuery query = em.createQuery(cq);
            if (this.offset > 0) {
                query.setFirstResult(this.offset);
            }
            if (this.limit < Integer.MAX_VALUE) {
                query.setMaxResults(this.limit);
            }
            return query.getResultList();
        }

        private void applyPredicates(CriteriaBuilder cb, CriteriaQuery<?> cq, Root<E> root) {
            if (this.predicates.isEmpty()) {
                return;
            }
            ArrayList<Predicate> jpaPredicates = new ArrayList<Predicate>();
            for (PredicateSpec spec : this.predicates) {
                Path<?> path = this.resolvePath(root, spec.attribute);
                Predicate predicate = switch (spec.type.ordinal()) {
                    default -> throw new MatchException(null, null);
                    case 0 -> cb.equal(path, spec.value);
                    case 1 -> cb.like(path, (String)spec.value);
                    case 2 -> path.in((Collection)spec.value);
                    case 3 -> cb.isNotNull(path);
                    case 4 -> cb.isNull(path);
                    case 5 -> cb.greaterThan(path, (Comparable)spec.value);
                    case 6 -> cb.lessThan(path, (Comparable)spec.value);
                };
                jpaPredicates.add(predicate);
            }
            cq.where((Predicate[])jpaPredicates.toArray(Predicate[]::new));
        }

        private Path<?> resolvePath(Root<E> root, String attribute) {
            if (!attribute.contains(".")) {
                return root.get(attribute);
            }
            String[] parts = attribute.split("\\.");
            Path path = root.get(parts[0]);
            for (int i = 1; i < parts.length; ++i) {
                path = path.get(parts[i]);
            }
            return path;
        }

        private record PredicateSpec(String attribute, Object value, PredicateType type) {
        }

        private static enum PredicateType {
            EQUAL,
            LIKE,
            IN,
            NOT_NULL,
            IS_NULL,
            GREATER_THAN,
            LESS_THAN;

        }
    }
}

