/*
 * Decompiled with CFR 0.152.
 */
package de.jexcellence.dependency.downloader;

import de.jexcellence.dependency.model.DependencyCoordinate;
import de.jexcellence.dependency.model.DownloadResult;
import de.jexcellence.dependency.repository.RepositoryType;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.jar.JarFile;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class DependencyDownloader {
    private static final String LOGGER_NAME = "JExDependency";
    private static final String USER_AGENT = "JEDependency-Downloader/2.0.0";
    private static final String ACCEPT = "application/java-archive, application/octet-stream, */*;q=0.1";
    private static final int CONNECTION_TIMEOUT_MS = 10000;
    private static final int READ_TIMEOUT_MS = 30000;
    private static final int BUFFER_SIZE = 8192;
    private static final int MAX_REDIRECTS = 5;
    private static final long MIN_JAR_SIZE = 1024L;
    private final Logger logger = Logger.getLogger("JExDependency");
    private final List<String> customRepositories = new ArrayList<String>();
    private final ExecutorService executorService = Executors.newVirtualThreadPerTaskExecutor();

    public void addRepository(@NotNull String repositoryUrl) {
        Object normalizedUrl = repositoryUrl.endsWith("/") ? repositoryUrl : repositoryUrl + "/";
        this.customRepositories.add((String)normalizedUrl);
        this.logger.fine("Added custom repository: " + (String)normalizedUrl);
    }

    @NotNull
    public CompletableFuture<DownloadResult> downloadAsync(@NotNull DependencyCoordinate coordinate, @NotNull File targetDirectory) {
        return CompletableFuture.supplyAsync(() -> this.download(coordinate, targetDirectory), this.executorService);
    }

    @NotNull
    public DownloadResult download(@NotNull DependencyCoordinate coordinate, @NotNull File targetDirectory) {
        File targetFile = new File(targetDirectory, coordinate.toFileName());
        if (this.isValidExistingFile(targetFile)) {
            this.logger.fine("Dependency already exists: " + targetFile.getName());
            return DownloadResult.success(coordinate, targetFile);
        }
        this.logger.fine("Downloading dependency: " + coordinate.toGavString());
        for (String customRepo : this.customRepositories) {
            String downloadUrl = customRepo + coordinate.toRepositoryPath();
            this.logger.finest("Trying custom repository: " + downloadUrl);
            if (!this.attemptDownload(downloadUrl, targetFile)) continue;
            this.logger.fine("Downloaded from custom repository");
            return DownloadResult.success(coordinate, targetFile);
        }
        for (RepositoryType repository : RepositoryType.values()) {
            String downloadUrl = repository.buildUrl(coordinate);
            this.logger.finest("Trying repository: " + repository.name() + " at " + downloadUrl);
            if (!this.attemptDownload(downloadUrl, targetFile)) continue;
            this.logger.fine("Downloaded from repository: " + repository.name());
            return DownloadResult.success(coordinate, targetFile);
        }
        String errorMessage = "Failed to download from any repository";
        this.logger.warning("Failed to download from any repository: " + coordinate.toGavString());
        return DownloadResult.failure(coordinate, "Failed to download from any repository");
    }

    private boolean isValidExistingFile(@NotNull File file) {
        return file.isFile() && file.length() > 0L && this.isValidJarFile(file);
    }

    private boolean attemptDownload(@NotNull String downloadUrl, @NotNull File targetFile) {
        try {
            URI uri = URI.create(downloadUrl);
            URL url = uri.toURL();
            for (int redirectCount = 0; redirectCount <= 5; ++redirectCount) {
                HttpURLConnection connection = this.createConnection(url);
                connection.setInstanceFollowRedirects(false);
                int responseCode = connection.getResponseCode();
                if (responseCode >= 200 && responseCode < 300) {
                    return this.handleSuccessfulResponse(connection, url, targetFile);
                }
                if (responseCode >= 300 && responseCode < 400) {
                    String location = connection.getHeaderField("Location");
                    if (location == null || location.isEmpty()) {
                        this.logger.warning("Redirect without Location header from: " + String.valueOf(url));
                        return false;
                    }
                    url = URI.create(location).toURL();
                    this.logger.finest("Redirect " + responseCode + " to " + String.valueOf(url));
                    continue;
                }
                if (responseCode != 404) {
                    this.logger.warning("HTTP " + responseCode + " when downloading " + String.valueOf(url));
                }
                return false;
            }
            this.logger.warning("Too many redirects (5) for " + downloadUrl);
            return false;
        }
        catch (Exception exception) {
            this.logger.log(Level.FINE, "Download failed from URL: " + downloadUrl, exception);
            return false;
        }
    }

    private boolean handleSuccessfulResponse(@NotNull HttpURLConnection connection, @NotNull URL url, @NotNull File targetFile) throws IOException {
        long bytesWritten;
        long contentLength = this.parseContentLength(connection.getHeaderField("Content-Length"));
        String contentType = this.safeLowerCase(connection.getHeaderField("Content-Type"));
        File tempFile = new File(targetFile.getParentFile(), targetFile.getName() + ".part");
        Files.createDirectories(targetFile.getParentFile().toPath(), new FileAttribute[0]);
        try (InputStream inputStream = connection.getInputStream();
             FileOutputStream outputStream = new FileOutputStream(tempFile);){
            bytesWritten = this.transferData(inputStream, outputStream);
        }
        if (!this.validateDownload(tempFile, bytesWritten, contentLength, contentType, url)) {
            this.safeDelete(tempFile);
            return false;
        }
        this.moveToFinalLocation(tempFile, targetFile);
        this.logger.fine("Downloaded: " + targetFile.getName() + " (" + bytesWritten + " bytes)");
        return true;
    }

    private boolean validateDownload(@NotNull File file, long bytesWritten, long expectedLength, @Nullable String contentType, @NotNull URL url) {
        if (bytesWritten <= 0L) {
            this.logger.warning("Downloaded 0 bytes from " + String.valueOf(url));
            return false;
        }
        if (expectedLength > 0L && bytesWritten != expectedLength) {
            this.logger.warning(String.format(Locale.ROOT, "Content-Length mismatch for %s: expected %d, got %d", url, expectedLength, bytesWritten));
            return false;
        }
        if (!this.isValidJarFile(file)) {
            this.logger.warning("Downloaded file is not a valid JAR: " + file.getName() + " (Content-Type=" + contentType + ", bytes=" + bytesWritten + ")");
            return false;
        }
        return true;
    }

    private void moveToFinalLocation(@NotNull File source, @NotNull File destination) throws IOException {
        try {
            Files.move(source.toPath(), destination.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AtomicMoveNotSupportedException exception) {
            Files.move(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
    }

    @NotNull
    private HttpURLConnection createConnection(@NotNull URL url) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(30000);
        connection.setRequestProperty("User-Agent", USER_AGENT);
        connection.setRequestProperty("Accept", ACCEPT);
        return connection;
    }

    private long transferData(@NotNull InputStream inputStream, @NotNull FileOutputStream outputStream) throws IOException {
        int bytesRead;
        byte[] buffer = new byte[8192];
        long totalBytes = 0L;
        while ((bytesRead = inputStream.read(buffer)) != -1) {
            if (bytesRead <= 0) continue;
            outputStream.write(buffer, 0, bytesRead);
            totalBytes += (long)bytesRead;
        }
        return totalBytes;
    }

    private boolean isValidJarFile(@NotNull File file) {
        boolean bl;
        if (!file.isFile() || file.length() < 1024L) {
            return false;
        }
        JarFile jarFile = new JarFile(file, true);
        try {
            bl = jarFile.entries().hasMoreElements();
        }
        catch (Throwable throwable) {
            try {
                try {
                    jarFile.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception exception) {
                return false;
            }
        }
        jarFile.close();
        return bl;
    }

    private void safeDelete(@NotNull File file) {
        try {
            Files.deleteIfExists(file.toPath());
        }
        catch (IOException exception) {
            this.logger.log(Level.FINE, "Failed to delete file: " + String.valueOf(file), exception);
        }
    }

    private long parseContentLength(@Nullable String header) {
        if (header == null) {
            return -1L;
        }
        try {
            return Long.parseLong(header.trim());
        }
        catch (NumberFormatException exception) {
            return -1L;
        }
    }

    @Nullable
    private String safeLowerCase(@Nullable String value) {
        return value == null ? null : value.toLowerCase(Locale.ROOT);
    }

    public void shutdown() {
        this.executorService.shutdown();
    }
}

