/*
 * Decompiled with CFR 0.152.
 */
package net.godlycow.org.commands.impl;

import java.time.LocalDate;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import net.godlycow.org.commands.config.NickConfig;
import net.godlycow.org.language.LanguageManager;
import net.godlycow.org.language.PlayerLanguageManager;
import net.godlycow.org.nick.NickManager;
import net.godlycow.org.nick.storage.NickStorage;
import org.bukkit.Bukkit;
import org.bukkit.command.Command;
import org.bukkit.command.CommandExecutor;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Player;

public final class NickCommand
implements CommandExecutor {
    private static final String PERMISSION = "essc.command.nick";
    private static final String ADMIN_PERMISSION = "essc.command.nicks";
    private final NickConfig config;
    private final PlayerLanguageManager langManager;
    private final NickStorage nickStorage;
    private final NickManager nickManager;

    public NickCommand(PlayerLanguageManager langManager, NickConfig config, NickStorage nickStorage, NickManager nickManager) {
        this.langManager = langManager;
        this.config = config;
        this.nickStorage = nickStorage;
        this.nickManager = nickManager;
    }

    public boolean onCommand(CommandSender sender, Command command, String label, String[] args) {
        if (!this.config.enabled) {
            sender.sendMessage(this.langManager.getMessageFor(null, "commands.nick.disabled", "<red>The nickname system is currently disabled.", new LanguageManager.ComponentPlaceholder[0]).toString());
            return true;
        }
        if (command.getName().equalsIgnoreCase("nick")) {
            return this.handleNick(sender, args);
        }
        if (command.getName().equalsIgnoreCase("nicks")) {
            return this.handleNicks(sender, args);
        }
        return false;
    }

    private boolean handleNick(CommandSender sender, String[] args) {
        if (!(sender instanceof Player)) {
            sender.sendMessage(this.langManager.getMessageFor(null, "commands.nick.only-player", "<red>Only players can use this command.", new LanguageManager.ComponentPlaceholder[0]).toString());
            return true;
        }
        Player player = (Player)sender;
        if (!player.hasPermission(PERMISSION)) {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.no-permission", "<red>You need permission <yellow>{permission}</yellow>!", LanguageManager.ComponentPlaceholder.of("{permission}", PERMISSION)));
            return true;
        }
        if (args.length < 1) {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.usage", "<yellow>Usage: <white>/nick <nickname> or /nick reset", new LanguageManager.ComponentPlaceholder[0]).toString());
            return true;
        }
        if (this.config.allowReset && args[0].equalsIgnoreCase("reset")) {
            return this.resetOwnNick(player);
        }
        return this.setNick(player, String.join((CharSequence)" ", args));
    }

    private boolean setNick(Player player, String newNick) {
        if (!this.isValidNick(player, newNick)) {
            return true;
        }
        UUID uuid = player.getUniqueId();
        this.nickStorage.getNickname(uuid).thenAccept(opt -> {
            opt.ifPresent(existingNick -> {
                if (this.config.cooldown > 0) {
                    long l = System.currentTimeMillis() - (long)(this.config.cooldown * 1000);
                }
            });
            if (this.config.maxChangesPerDay > 0) {
                String today = LocalDate.now().toString();
                this.nickStorage.getDailyChanges(uuid, today).thenAccept(changesOpt -> {
                    int changes = changesOpt.orElse(0);
                    if (changes >= this.config.maxChangesPerDay) {
                        player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.daily-limit", "<red>You can only change your nickname <yellow>{limit}</yellow> times per day.", LanguageManager.ComponentPlaceholder.of("{limit}", this.config.maxChangesPerDay)));
                        return;
                    }
                    this.proceedWithNickChange(player, newNick, today, changes);
                });
            } else {
                this.proceedWithNickChange(player, newNick, null, 0);
            }
        });
        return true;
    }

    private void proceedWithNickChange(Player player, String newNick, String day, int dailyChanges) {
        UUID uuid = player.getUniqueId();
        ((CompletableFuture)this.nickStorage.setNickname(uuid, newNick).thenAccept(v -> {
            this.nickManager.applyNick(uuid, newNick);
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.set", "<green>Your nickname has been set to <white>{nick}</white>.", LanguageManager.ComponentPlaceholder.of("{nick}", newNick)));
            if (day != null) {
                this.nickStorage.incrementDailyChanges(uuid);
            }
        })).exceptionally(ex -> {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.error", "<red>Failed to save nickname. Please try again.", new LanguageManager.ComponentPlaceholder[0]));
            return null;
        });
    }

    private boolean resetOwnNick(Player player) {
        UUID uuid = player.getUniqueId();
        ((CompletableFuture)this.nickStorage.removeNickname(uuid).thenAccept(v -> {
            this.nickManager.removeNick(uuid);
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.reset-self", "<green>Your nickname has been reset.", new LanguageManager.ComponentPlaceholder[0]));
        })).exceptionally(ex -> {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.error", "<red>Failed to reset nickname. Please try again.", new LanguageManager.ComponentPlaceholder[0]));
            return null;
        });
        return true;
    }

    private boolean handleNicks(CommandSender sender, String[] args) {
        if (!sender.hasPermission(ADMIN_PERMISSION)) {
            Player p;
            sender.sendMessage(this.langManager.getMessageFor(sender instanceof Player ? (p = (Player)sender) : null, "commands.nicks.no-permission", "<red>You need permission <yellow>{permission}</yellow>!", LanguageManager.ComponentPlaceholder.of("{permission}", ADMIN_PERMISSION)).toString());
            return true;
        }
        if (args.length == 0) {
            sender.sendMessage(this.langManager.getMessageFor(null, "commands.nicks.usage", "<yellow>Usage: <white>/nicks reload|reset <player>", new LanguageManager.ComponentPlaceholder[0]).toString());
            return true;
        }
        if (args[0].equalsIgnoreCase("reload")) {
            return this.reloadNicks(sender);
        }
        if (args[0].equalsIgnoreCase("reset") && args.length == 2) {
            return this.resetOtherNick(sender, args[1]);
        }
        return false;
    }

    private boolean reloadNicks(CommandSender sender) {
        Player p;
        this.config.reload();
        this.nickManager.reloadAllNicks();
        sender.sendMessage(this.langManager.getMessageFor(sender instanceof Player ? (p = (Player)sender) : null, "commands.nicks.reload", "<green>Nickname system reloaded successfully.", new LanguageManager.ComponentPlaceholder[0]));
        return true;
    }

    private boolean resetOtherNick(CommandSender sender, String targetName) {
        Player target = Bukkit.getPlayer((String)targetName);
        if (target == null) {
            Player p;
            sender.sendMessage(this.langManager.getMessageFor(sender instanceof Player ? (p = (Player)sender) : null, "commands.nicks.not-found", "<red>Player <yellow>{player}</yellow> not found.", LanguageManager.ComponentPlaceholder.of("{player}", targetName)).toString());
            return true;
        }
        UUID uuid = target.getUniqueId();
        ((CompletableFuture)this.nickStorage.removeNickname(uuid).thenAccept(v -> {
            Player p;
            this.nickManager.removeNick(uuid);
            sender.sendMessage(this.langManager.getMessageFor(sender instanceof Player ? (p = (Player)sender) : null, "commands.nicks.reset", "<green>Reset nickname for <yellow>{player}</yellow>.", LanguageManager.ComponentPlaceholder.of("{player}", target.getName())));
        })).exceptionally(ex -> {
            Player p;
            sender.sendMessage(this.langManager.getMessageFor(sender instanceof Player ? (p = (Player)sender) : null, "commands.nicks.error", "<red>Failed to reset nickname. Please try again.", new LanguageManager.ComponentPlaceholder[0]));
            return null;
        });
        return true;
    }

    private boolean isValidNick(Player player, String nick) {
        String stripped = this.stripFormatting(nick);
        if (stripped.length() < this.config.minLength || stripped.length() > this.config.maxLength) {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.invalid-length", "<red>Nickname must be between <yellow>{min}</yellow> and <yellow>{max}</yellow> characters.", LanguageManager.ComponentPlaceholder.of("{min}", this.config.minLength), LanguageManager.ComponentPlaceholder.of("{max}", this.config.maxLength)));
            return false;
        }
        for (String string : this.config.blockedWords) {
            if (!stripped.toLowerCase().contains(string.toLowerCase())) continue;
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.blocked-word", "<red>The word <yellow>{word}</yellow> is not allowed.", LanguageManager.ComponentPlaceholder.of("{word}", string)));
            return false;
        }
        for (String string : this.config.blacklistPatterns) {
            try {
                if (!Pattern.compile(string, 2).matcher(stripped).find()) continue;
                player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.blacklisted", "<red>This nickname format is not allowed.", new LanguageManager.ComponentPlaceholder[0]));
                return false;
            }
            catch (Exception exception) {
            }
        }
        if (!this.config.allowFormatting && this.containsFormatting(nick)) {
            player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.no-formatting", "<red>Formatting is not allowed in nicknames.", new LanguageManager.ComponentPlaceholder[0]));
            return false;
        }
        if (!this.config.allowDuplicates) {
            for (Map.Entry entry : this.nickManager.getAllCachedNicks().entrySet()) {
                if (!((String)entry.getValue()).equalsIgnoreCase(nick) || ((UUID)entry.getKey()).equals(player.getUniqueId())) continue;
                player.sendMessage(this.langManager.getMessageFor(player, "commands.nick.duplicate", "<red>This nickname is already taken by another player.", new LanguageManager.ComponentPlaceholder[0]));
                return false;
            }
        }
        return true;
    }

    private String stripFormatting(String input) {
        return input.replaceAll("<[^>]+>", "");
    }

    private boolean containsFormatting(String input) {
        return input.contains("<") && input.contains(">");
    }
}

