/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.ezcountdown.storage;

import com.skyblockexp.ezcountdown.countdown.Countdown;
import com.skyblockexp.ezcountdown.countdown.CountdownType;
import com.skyblockexp.ezcountdown.display.DisplayType;
import com.skyblockexp.ezcountdown.manager.CountdownDefaults;
import com.skyblockexp.ezcountdown.storage.CountdownStorage;
import com.skyblockexp.ezcountdown.util.DurationParser;
import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.logging.Level;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.plugin.java.JavaPlugin;

public final class YamlCountdownStorage
implements CountdownStorage {
    private static final DateTimeFormatter DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
    private final JavaPlugin plugin;
    private final CountdownDefaults defaults;
    private final File storageFile;

    public YamlCountdownStorage(JavaPlugin plugin, CountdownDefaults defaults, File storageFile) {
        this.plugin = Objects.requireNonNull(plugin, "plugin");
        this.defaults = Objects.requireNonNull(defaults, "defaults");
        this.storageFile = Objects.requireNonNull(storageFile, "storageFile");
    }

    @Override
    public Collection<Countdown> loadCountdowns() {
        if (!this.storageFile.exists()) {
            this.plugin.saveResource(this.storageFile.getName(), false);
        }
        YamlConfiguration config = YamlConfiguration.loadConfiguration((File)this.storageFile);
        ConfigurationSection root = config.getConfigurationSection("countdowns");
        ArrayList<Countdown> countdowns = new ArrayList<Countdown>();
        if (root == null) {
            return countdowns;
        }
        for (String key : root.getKeys(false)) {
            ConfigurationSection section = root.getConfigurationSection(key);
            if (section == null) continue;
            try {
                Countdown countdown = this.parseCountdown(key, section);
                if (countdown == null) continue;
                countdowns.add(countdown);
            }
            catch (IllegalArgumentException ex) {
                this.plugin.getLogger().log(Level.WARNING, "Failed to load countdown " + key + ": " + ex.getMessage(), ex);
            }
        }
        return countdowns;
    }

    @Override
    public void saveCountdowns(Collection<Countdown> countdowns) {
        YamlConfiguration config = new YamlConfiguration();
        ConfigurationSection root = config.createSection("countdowns");
        for (Countdown countdown : countdowns) {
            ConfigurationSection section = root.createSection(countdown.getName());
            section.set("type", (Object)countdown.getType().name());
            section.set("running", (Object)countdown.isRunning());
            section.set("display.types", countdown.getDisplayTypes().stream().map(Enum::name).toList());
            section.set("display.update-interval", (Object)countdown.getUpdateIntervalSeconds());
            if (countdown.getVisibilityPermission() == null || countdown.getVisibilityPermission().isBlank()) {
                section.set("display.visibility", (Object)"all");
            } else {
                section.set("display.visibility", (Object)countdown.getVisibilityPermission());
            }
            section.set("messages.format", (Object)countdown.getFormatMessage());
            section.set("messages.start", (Object)countdown.getStartMessage());
            section.set("messages.end", (Object)countdown.getEndMessage());
            section.set("commands.end", countdown.getEndCommands());
            section.set("zone", (Object)countdown.getZoneId().getId());
            switch (countdown.getType()) {
                case FIXED_DATE: {
                    section.set("target", (Object)DATE_TIME_FORMAT.format(countdown.getTargetInstant().atZone(countdown.getZoneId())));
                    break;
                }
                case DURATION: 
                case MANUAL: {
                    section.set("duration", (Object)(countdown.getDurationSeconds() + "s"));
                    break;
                }
                case RECURRING: {
                    section.set("recurring.month", (Object)countdown.getRecurringMonth());
                    section.set("recurring.day", (Object)countdown.getRecurringDay());
                    section.set("recurring.time", (Object)countdown.getRecurringTime().toString());
                }
            }
        }
        try {
            config.save(this.storageFile);
        }
        catch (IOException ex) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to save countdowns.yml", ex);
        }
    }

    private Countdown parseCountdown(String name, ConfigurationSection section) {
        CountdownType type = CountdownType.valueOf(section.getString("type", "FIXED_DATE").toUpperCase(Locale.ROOT));
        EnumSet<DisplayType> displayTypes = this.parseDisplayTypes(section.getStringList("display.types"));
        int updateInterval = section.getInt("display.update-interval", this.defaults.updateIntervalSeconds());
        String visibility = section.getString("display.visibility", this.defaults.visibilityPermission());
        if ("all".equalsIgnoreCase(visibility)) {
            visibility = null;
        }
        String format = section.getString("messages.format", this.defaults.formatMessage());
        String start = section.getString("messages.start", this.defaults.startMessage());
        String end = section.getString("messages.end", this.defaults.endMessage());
        List<String> endCommands = section.getStringList("commands.end").stream().filter(command -> command != null && !command.isBlank()).toList();
        ZoneId zoneId = ZoneId.of(section.getString("zone", this.defaults.zoneId().getId()));
        Countdown countdown = new Countdown(name, type, displayTypes, updateInterval, visibility, format, start, end, endCommands, zoneId);
        countdown.setRunning(section.getBoolean("running", type != CountdownType.MANUAL));
        switch (type) {
            case FIXED_DATE: {
                String target = section.getString("target");
                if (target == null) {
                    throw new IllegalArgumentException("Missing target date for fixed date countdown.");
                }
                countdown.setTargetInstant(ZonedDateTime.of(LocalDateTime.parse(target, DATE_TIME_FORMAT), zoneId).toInstant());
                break;
            }
            case DURATION: 
            case MANUAL: {
                String durationValue = section.getString("duration", "0s");
                countdown.setDurationSeconds(DurationParser.parseToSeconds(durationValue));
                if (!countdown.isRunning() || countdown.getTargetInstant() != null) break;
                countdown.setTargetInstant(Instant.now().plusSeconds(countdown.getDurationSeconds()));
                break;
            }
            case RECURRING: {
                countdown.setRecurringMonth(section.getInt("recurring.month", 1));
                countdown.setRecurringDay(section.getInt("recurring.day", 1));
                String timeValue = section.getString("recurring.time", "00:00");
                countdown.setRecurringTime(LocalTime.parse(timeValue));
                countdown.setTargetInstant(countdown.resolveNextRecurringTarget(Instant.now()));
            }
        }
        return countdown;
    }

    private EnumSet<DisplayType> parseDisplayTypes(List<String> entries) {
        EnumSet<DisplayType> types = EnumSet.noneOf(DisplayType.class);
        if (entries == null || entries.isEmpty()) {
            types.addAll(this.defaults.displayTypes());
            return types;
        }
        for (String entry : entries) {
            try {
                types.add(DisplayType.valueOf(entry.toUpperCase(Locale.ROOT)));
            }
            catch (IllegalArgumentException ex) {
                this.plugin.getLogger().warning("Unknown display type: " + entry);
            }
        }
        if (types.isEmpty()) {
            types.addAll(this.defaults.displayTypes());
        }
        return types;
    }
}

