/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.ezcountdown.manager;

import com.skyblockexp.ezcountdown.EzCountdownPlugin;
import com.skyblockexp.ezcountdown.countdown.Countdown;
import com.skyblockexp.ezcountdown.countdown.CountdownType;
import com.skyblockexp.ezcountdown.display.DisplayManager;
import com.skyblockexp.ezcountdown.event.CountdownEndEvent;
import com.skyblockexp.ezcountdown.event.CountdownStartEvent;
import com.skyblockexp.ezcountdown.event.CountdownTickEvent;
import com.skyblockexp.ezcountdown.firework.FireworkShowManager;
import com.skyblockexp.ezcountdown.location.LocationManager;
import com.skyblockexp.ezcountdown.manager.MessageManager;
import com.skyblockexp.ezcountdown.storage.CountdownStorage;
import com.skyblockexp.ezcountdown.util.DiscordWebhookConfig;
import com.skyblockexp.ezcountdown.util.DiscordWebhookSender;
import com.skyblockexp.ezcountdown.util.TimeFormat;
import java.io.File;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.LongAdder;
import java.util.logging.Level;
import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.command.CommandSender;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitTask;

public final class CountdownManager {
    private final JavaPlugin plugin;
    private final DiscordWebhookConfig discordWebhookConfig;
    private final CountdownStorage storage;
    private final DisplayManager displayManager;
    private final MessageManager messageManager;
    private final Map<String, Countdown> countdowns = new ConcurrentHashMap<String, Countdown>();
    private final Map<String, Instant> lastUpdate = new HashMap<String, Instant>();
    private final LongAdder executedCount = new LongAdder();
    private final LocationManager locationManager;
    private final FireworkShowManager fireworkShowManager = new FireworkShowManager();
    private BukkitTask task;

    public CountdownManager(JavaPlugin plugin, CountdownStorage storage, DisplayManager displayManager, MessageManager messageManager, LocationManager locationManager) {
        this.plugin = Objects.requireNonNull(plugin, "plugin");
        this.storage = Objects.requireNonNull(storage, "storage");
        this.displayManager = Objects.requireNonNull(displayManager, "displayManager");
        this.messageManager = Objects.requireNonNull(messageManager, "messageManager");
        this.locationManager = Objects.requireNonNull(locationManager, "locationManager");
        if (plugin instanceof EzCountdownPlugin) {
            EzCountdownPlugin ezPlugin = (EzCountdownPlugin)plugin;
            this.discordWebhookConfig = ezPlugin.getDiscordWebhookConfig();
        } else {
            this.discordWebhookConfig = null;
        }
    }

    public void load() {
        this.countdowns.clear();
        this.lastUpdate.clear();
        for (Countdown countdown : this.storage.loadCountdowns()) {
            this.countdowns.put(this.normalizeName(countdown.getName()), countdown);
        }
        this.startTask();
    }

    public void shutdown() {
        this.stopTask();
        this.displayManager.clearAll();
    }

    public Collection<Countdown> getCountdowns() {
        return Collections.unmodifiableCollection(this.countdowns.values());
    }

    public int getCountdownCount() {
        return this.countdowns.size();
    }

    public int getExecutedCount() {
        long value = this.executedCount.sum();
        return value > Integer.MAX_VALUE ? Integer.MAX_VALUE : (int)value;
    }

    public Optional<Countdown> getCountdown(String name) {
        return Optional.ofNullable(this.countdowns.get(this.normalizeName(name)));
    }

    public boolean createCountdown(Countdown countdown) {
        String key = this.normalizeName(countdown.getName());
        if (this.countdowns.containsKey(key)) {
            return false;
        }
        this.countdowns.put(key, countdown);
        if (countdown.isRunning()) {
            if (countdown.getType() == CountdownType.DURATION || countdown.getType() == CountdownType.MANUAL) {
                countdown.setTargetInstant(Instant.now().plusSeconds(countdown.getDurationSeconds()));
            }
            this.fireStart(countdown);
        }
        return true;
    }

    public boolean deleteCountdown(String name) {
        Countdown removed = this.countdowns.remove(this.normalizeName(name));
        if (removed != null) {
            this.displayManager.clearCountdown(removed);
            return true;
        }
        return false;
    }

    public boolean startCountdown(String name) {
        Countdown countdown = this.countdowns.get(this.normalizeName(name));
        if (countdown == null) {
            return false;
        }
        if (countdown.isRunning()) {
            return true;
        }
        countdown.setRunning(true);
        if (countdown.getType() == CountdownType.DURATION || countdown.getType() == CountdownType.MANUAL) {
            countdown.setTargetInstant(Instant.now().plusSeconds(countdown.getDurationSeconds()));
        } else if (countdown.getType() == CountdownType.RECURRING) {
            countdown.setTargetInstant(countdown.resolveNextRecurringTarget(Instant.now()));
        }
        this.fireStart(countdown);
        return true;
    }

    public boolean stopCountdown(String name) {
        Countdown countdown = this.countdowns.get(this.normalizeName(name));
        if (countdown == null) {
            return false;
        }
        countdown.setRunning(false);
        if (countdown.getType() == CountdownType.DURATION || countdown.getType() == CountdownType.MANUAL) {
            countdown.setTargetInstant(null);
        }
        this.displayManager.clearCountdown(countdown);
        return true;
    }

    public void save() {
        this.storage.saveCountdowns(this.countdowns.values());
    }

    private void startTask() {
        this.stopTask();
        this.task = Bukkit.getScheduler().runTaskTimer((Plugin)this.plugin, this::tick, 20L, 20L);
    }

    private void stopTask() {
        if (this.task != null) {
            this.task.cancel();
            this.task = null;
        }
    }

    private void tick() {
        Instant now = Instant.now();
        for (Countdown countdown : this.countdowns.values()) {
            if (!countdown.isRunning()) continue;
            Instant target = countdown.getTargetInstant();
            if (target == null) {
                if (countdown.getType() == CountdownType.DURATION || countdown.getType() == CountdownType.MANUAL) {
                    countdown.setTargetInstant(now.plusSeconds(countdown.getDurationSeconds()));
                } else {
                    if (countdown.getType() != CountdownType.RECURRING) continue;
                    countdown.setTargetInstant(countdown.resolveNextRecurringTarget(now));
                }
                target = countdown.getTargetInstant();
            }
            long remaining = Math.max(0L, target.getEpochSecond() - now.getEpochSecond());
            Instant last = this.lastUpdate.getOrDefault(countdown.getName(), Instant.EPOCH);
            if (now.isAfter(last.plusSeconds(countdown.getUpdateIntervalSeconds()))) {
                this.lastUpdate.put(countdown.getName(), now);
                String message = this.buildMessage(countdown, remaining);
                this.displayManager.display(countdown, message, remaining);
                this.fireTick(countdown, remaining);
            }
            if (remaining > 0L) continue;
            this.fireEnd(countdown);
            if (countdown.getType() == CountdownType.RECURRING) {
                countdown.setTargetInstant(countdown.resolveNextRecurringTarget(now));
                continue;
            }
            countdown.setRunning(false);
            this.displayManager.clearCountdown(countdown);
        }
    }

    private String buildMessage(Countdown countdown, long remaining) {
        TimeFormat.TimeParts parts = TimeFormat.toParts(remaining);
        String formatted = TimeFormat.format(parts);
        String message = countdown.getFormatMessage();
        message = message.replace("{name}", countdown.getName()).replace("{days}", String.valueOf(parts.days())).replace("{hours}", String.valueOf(parts.hours())).replace("{minutes}", String.valueOf(parts.minutes())).replace("{seconds}", String.valueOf(parts.seconds())).replace("{formatted}", formatted);
        return this.messageManager.formatWithPrefix(message, Map.of());
    }

    private void fireStart(Countdown countdown) {
        Location loc;
        FireworkConfig fwConfig;
        String teleportLocation;
        String message = countdown.getStartMessage();
        if (message != null && !message.isBlank()) {
            this.displayManager.broadcastMessage(this.messageManager.formatWithPrefix(message, Map.of("name", countdown.getName())));
        }
        if ((teleportLocation = this.getTeleportLocation(countdown, "start")) != null) {
            for (Player player : Bukkit.getOnlinePlayers()) {
                this.locationManager.teleportPlayer(player, teleportLocation);
            }
        }
        if ((fwConfig = this.getFireworkConfig(countdown, "start")) != null && (loc = this.getLocationFromName(fwConfig.location)) != null) {
            this.fireworkShowManager.launchCircleFireworkShow(loc.getWorld(), loc, fwConfig.color, fwConfig.power, fwConfig.count, fwConfig.rows, fwConfig.interval);
        }
        this.sendDiscordWebhooks("countdown_start", countdown, null);
        try {
            Bukkit.getPluginManager().callEvent((Event)new CountdownStartEvent(countdown));
        }
        catch (Exception ex) {
            this.plugin.getLogger().log(Level.WARNING, "Error while firing CountdownStartEvent", ex);
        }
    }

    private void fireTick(Countdown countdown, long remaining) {
        try {
            Bukkit.getPluginManager().callEvent((Event)new CountdownTickEvent(countdown, remaining));
        }
        catch (Exception ex) {
            this.plugin.getLogger().log(Level.WARNING, "Error while firing CountdownTickEvent", ex);
        }
    }

    private void fireEnd(Countdown countdown) {
        Location loc;
        FireworkConfig fwConfig;
        String teleportLocation;
        String message = countdown.getEndMessage();
        if (message != null && !message.isBlank()) {
            this.displayManager.broadcastMessage(this.messageManager.formatWithPrefix(message, Map.of("name", countdown.getName())));
        }
        if ((teleportLocation = this.getTeleportLocation(countdown, "end")) != null) {
            for (Player player : Bukkit.getOnlinePlayers()) {
                this.locationManager.teleportPlayer(player, teleportLocation);
            }
        }
        if ((fwConfig = this.getFireworkConfig(countdown, "end")) != null && (loc = this.getLocationFromName(fwConfig.location)) != null) {
            this.fireworkShowManager.launchCircleFireworkShow(loc.getWorld(), loc, fwConfig.color, fwConfig.power, fwConfig.count, fwConfig.rows, fwConfig.interval);
        }
        this.executeEndCommands(countdown);
        this.executedCount.increment();
        this.sendDiscordWebhooks("countdown_end", countdown, null);
        try {
            Bukkit.getPluginManager().callEvent((Event)new CountdownEndEvent(countdown));
        }
        catch (Exception ex) {
            this.plugin.getLogger().log(Level.WARNING, "Error while firing CountdownEndEvent", ex);
        }
    }

    private FireworkConfig getFireworkConfig(Countdown countdown, String phase) {
        try {
            YamlConfiguration config = YamlConfiguration.loadConfiguration((File)new File(this.plugin.getDataFolder(), "countdowns.yml"));
            String base = "countdowns." + countdown.getName() + ".firework." + phase;
            if (config.contains(base + ".location")) {
                FireworkConfig fw = new FireworkConfig();
                fw.location = config.getString(base + ".location");
                fw.color = config.getString(base + ".color", "WHITE");
                fw.power = config.getInt(base + ".power", 1);
                fw.count = config.getInt(base + ".count", 8);
                fw.rows = config.getInt(base + ".rows", 1);
                fw.interval = config.getInt(base + ".interval", 10);
                return fw;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private Location getLocationFromName(String name) {
        if (name == null) {
            return null;
        }
        YamlConfiguration config = YamlConfiguration.loadConfiguration((File)new File(this.plugin.getDataFolder(), "locations.yml"));
        if (!config.contains(name)) {
            return null;
        }
        String worldName = config.getString(name + ".world");
        World world = Bukkit.getWorld((String)worldName);
        if (world == null) {
            return null;
        }
        double x = config.getDouble(name + ".x");
        double y = config.getDouble(name + ".y");
        double z = config.getDouble(name + ".z");
        float yaw = (float)config.getDouble(name + ".yaw");
        float pitch = (float)config.getDouble(name + ".pitch");
        return new Location(world, x, y, z, yaw, pitch);
    }

    private String getTeleportLocation(Countdown countdown, String phase) {
        try {
            YamlConfiguration config = YamlConfiguration.loadConfiguration((File)new File(this.plugin.getDataFolder(), "countdowns.yml"));
            String path = "countdowns." + countdown.getName() + ".teleport." + phase;
            if (config.contains(path)) {
                return config.getString(path);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private void sendDiscordWebhooks(String trigger, Countdown countdown, Map<String, String> extraPlaceholders) {
        if (this.discordWebhookConfig == null) {
            return;
        }
        HashMap<String, String> placeholders = new HashMap<String, String>();
        placeholders.put("{event}", countdown.getName());
        placeholders.put("{trigger}", trigger);
        if (extraPlaceholders != null) {
            placeholders.putAll(extraPlaceholders);
        }
        for (DiscordWebhookConfig.Webhook webhook : this.discordWebhookConfig.getWebhooks()) {
            DiscordWebhookConfig.Embed embed;
            if (!webhook.enabled || webhook.url == null || webhook.url.isBlank() || webhook.triggers == null || webhook.triggers.isEmpty() || !webhook.triggers.contains(trigger) || (embed = webhook.embed) == null) continue;
            HashMap<String, Object> embedJson = new HashMap<String, Object>();
            embedJson.put("title", this.replacePlaceholders(embed.title, countdown, placeholders));
            embedJson.put("description", this.replacePlaceholders(embed.description, countdown, placeholders));
            if (embed.color != null && !embed.color.isBlank()) {
                try {
                    embedJson.put("color", Integer.decode(embed.color));
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (embed.footer != null) {
                HashMap<String, String> footer = new HashMap<String, String>();
                footer.put("text", this.replacePlaceholders(embed.footer.text, countdown, placeholders));
                if (embed.footer.iconUrl != null) {
                    footer.put("icon_url", embed.footer.iconUrl);
                }
                embedJson.put("footer", footer);
            }
            if (embed.thumbnailUrl != null && !embed.thumbnailUrl.isBlank()) {
                embedJson.put("thumbnail", Map.of("url", embed.thumbnailUrl));
            }
            if (embed.imageUrl != null && !embed.imageUrl.isBlank()) {
                embedJson.put("image", Map.of("url", embed.imageUrl));
            }
            if (embed.author != null) {
                HashMap<String, String> author = new HashMap<String, String>();
                author.put("name", this.replacePlaceholders(embed.author.name, countdown, placeholders));
                if (embed.author.iconUrl != null) {
                    author.put("icon_url", embed.author.iconUrl);
                }
                embedJson.put("author", author);
            }
            DiscordWebhookSender.sendWebhook(webhook.url, embedJson);
        }
    }

    private String replacePlaceholders(String input, Countdown countdown, Map<String, String> extra) {
        if (input == null) {
            return null;
        }
        String out = input.replace("{countdown}", countdown.getName());
        long remaining = 0L;
        if (countdown.getTargetInstant() != null) {
            remaining = Math.max(0L, countdown.getTargetInstant().getEpochSecond() - Instant.now().getEpochSecond());
        }
        TimeFormat.TimeParts parts = TimeFormat.toParts(remaining);
        out = out.replace("{time_left}", TimeFormat.format(parts));
        out = out.replace("{days}", String.valueOf(parts.days()));
        out = out.replace("{hours}", String.valueOf(parts.hours()));
        out = out.replace("{minutes}", String.valueOf(parts.minutes()));
        out = out.replace("{seconds}", String.valueOf(parts.seconds()));
        if (extra != null) {
            for (Map.Entry<String, String> e : extra.entrySet()) {
                out = out.replace(e.getKey(), e.getValue());
            }
        }
        return out;
    }

    private void executeEndCommands(Countdown countdown) {
        if (countdown.getEndCommands().isEmpty()) {
            return;
        }
        for (String command : countdown.getEndCommands()) {
            if (command == null || command.isBlank()) continue;
            String resolved = command.replace("{name}", countdown.getName());
            if (resolved.startsWith("/")) {
                resolved = resolved.substring(1);
            }
            try {
                Bukkit.dispatchCommand((CommandSender)Bukkit.getConsoleSender(), (String)resolved);
            }
            catch (Exception ex) {
                this.plugin.getLogger().log(Level.WARNING, "Error executing end command for countdown " + countdown.getName() + ": " + command, ex);
            }
        }
    }

    private String normalizeName(String name) {
        return name == null ? "" : name.toLowerCase(Locale.ROOT);
    }

    private static class FireworkConfig {
        String location;
        String color;
        int power;
        int count;
        int rows = 1;
        int interval = 10;

        private FireworkConfig() {
        }
    }
}

