/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.ezauction.storage.mysql;

import com.skyblockexp.ezauction.config.AuctionStorageConfiguration;
import com.skyblockexp.ezauction.storage.AuctionHistoryStorage;
import com.skyblockexp.ezauction.transaction.AuctionTransactionHistoryEntry;
import com.skyblockexp.ezauction.transaction.AuctionTransactionType;
import com.skyblockexp.ezauction.util.EconomyUtils;
import com.skyblockexp.ezauction.util.ItemStackSerialization;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bukkit.inventory.ItemStack;

public class MysqlAuctionHistoryStorage
implements AuctionHistoryStorage {
    private final Logger logger;
    private final AuctionStorageConfiguration.Mysql mysql;
    private final String jdbcUrl;
    private final String historyTable;
    private final Lock historyWriteLock = new ReentrantLock(true);
    private boolean driverLoaded = false;

    public MysqlAuctionHistoryStorage(Logger logger, AuctionStorageConfiguration.Mysql mysql) {
        this.logger = logger != null ? logger : Logger.getLogger(MysqlAuctionHistoryStorage.class.getName());
        this.mysql = mysql;
        this.jdbcUrl = "jdbc:mysql://" + mysql.host() + ":" + mysql.port() + "/" + mysql.database();
        String prefix = mysql.tablePrefix();
        this.historyTable = this.sanitize(prefix + "history");
    }

    private boolean isReady() {
        return this.driverLoaded;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean initialize() {
        if (this.driverLoaded) {
            return true;
        }
        try {
            Class.forName("com.skyblockexp.ezauction.libs.com.mysql.cj.jdbc.Driver");
            this.driverLoaded = true;
        }
        catch (ClassNotFoundException ex) {
            this.logger.log(Level.SEVERE, "MySQL JDBC driver not found.", ex);
            return false;
        }
        try (Connection connection = this.getConnection();){
            boolean bl;
            block17: {
                Statement statement = connection.createStatement();
                try {
                    statement.executeUpdate("CREATE TABLE IF NOT EXISTS `" + this.historyTable + "` (player_uuid CHAR(36) NOT NULL,entry_index INT NOT NULL,type VARCHAR(16) NOT NULL,timestamp BIGINT NOT NULL,price DOUBLE NOT NULL,counterpart_uuid CHAR(36) NULL,counterpart_name VARCHAR(64) NULL,item LONGTEXT NULL,PRIMARY KEY (player_uuid, entry_index)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
                    bl = true;
                    if (statement == null) break block17;
                }
                catch (Throwable throwable) {
                    if (statement != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                statement.close();
            }
            return bl;
        }
        catch (SQLException ex) {
            this.logger.log(Level.SEVERE, "Failed to create history table in MySQL.", ex);
            return false;
        }
    }

    @Override
    public Map<UUID, Deque<AuctionTransactionHistoryEntry>> loadAll() {
        HashMap<UUID, Deque<AuctionTransactionHistoryEntry>> result = new HashMap<UUID, Deque<AuctionTransactionHistoryEntry>>();
        if (!this.isReady()) {
            return result;
        }
        String query = "SELECT player_uuid, entry_index, type, timestamp, price, counterpart_uuid, counterpart_name, item FROM `" + this.historyTable + "` ORDER BY player_uuid, entry_index";
        try (Connection connection = this.getConnection();
             PreparedStatement statement = connection.prepareStatement(query);
             ResultSet resultSet = statement.executeQuery();){
            LinkedHashMap<UUID, Map> grouped = new LinkedHashMap<UUID, Map>();
            while (resultSet.next()) {
                UUID playerId = this.parseUuid(resultSet.getString("player_uuid"));
                if (playerId == null) continue;
                int index = resultSet.getInt("entry_index");
                AuctionTransactionType type = this.parseType(resultSet.getString("type"));
                if (type == null) continue;
                long timestamp = resultSet.getLong("timestamp");
                double price = EconomyUtils.normalizeCurrency(resultSet.getDouble("price"));
                UUID counterpartId = this.parseUuid(resultSet.getString("counterpart_uuid"));
                String counterpartName = resultSet.getString("counterpart_name");
                ItemStack item = ItemStackSerialization.deserialize(resultSet.getString("item"), this.logger);
                AuctionTransactionHistoryEntry entry = new AuctionTransactionHistoryEntry(type, counterpartId, counterpartName, price, timestamp, item);
                grouped.computeIfAbsent(playerId, key -> new LinkedHashMap()).put(index, entry);
            }
            for (Map.Entry entry : grouped.entrySet()) {
                Map ordered = (Map)entry.getValue();
                ArrayDeque deque = new ArrayDeque(ordered.size());
                ordered.entrySet().stream().sorted(Map.Entry.comparingByKey()).forEachOrdered(e -> deque.addLast((AuctionTransactionHistoryEntry)e.getValue()));
                result.put((UUID)entry.getKey(), deque);
            }
        }
        catch (SQLException ex) {
            this.logger.log(Level.SEVERE, "Failed to load transaction history from MySQL.", ex);
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void saveAll(Map<UUID, Deque<AuctionTransactionHistoryEntry>> history) {
        if (!this.isReady()) {
            return;
        }
        this.historyWriteLock.lock();
        String delete = "DELETE FROM `" + this.historyTable + "`";
        String insert = "INSERT INTO `" + this.historyTable + "` (player_uuid, entry_index, type, timestamp, price, counterpart_uuid, counterpart_name, item) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        try (Connection connection = this.getConnection();){
            connection.setAutoCommit(false);
            try (Statement deleteStatement = connection.createStatement();){
                deleteStatement.executeUpdate(delete);
            }
            try (PreparedStatement insertStatement = connection.prepareStatement(insert);){
                for (Map.Entry<UUID, Deque<AuctionTransactionHistoryEntry>> entry : history.entrySet()) {
                    this.writeHistoryEntries(insertStatement, entry.getKey(), entry.getValue());
                }
                insertStatement.executeBatch();
            }
            connection.commit();
        }
        catch (SQLException ex) {
            this.logger.log(Level.SEVERE, "Failed to save history to MySQL.", ex);
        }
        finally {
            this.historyWriteLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void savePlayerHistory(UUID playerId, Deque<AuctionTransactionHistoryEntry> history) {
        if (!this.isReady() || playerId == null) {
            return;
        }
        this.historyWriteLock.lock();
        String delete = "DELETE FROM `" + this.historyTable + "` WHERE player_uuid = ?";
        String insert = "INSERT INTO `" + this.historyTable + "` (player_uuid, entry_index, type, timestamp, price, counterpart_uuid, counterpart_name, item) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
        try (Connection connection = this.getConnection();){
            connection.setAutoCommit(false);
            try (PreparedStatement deleteStatement = connection.prepareStatement(delete);){
                deleteStatement.setString(1, playerId.toString());
                deleteStatement.executeUpdate();
            }
            try (PreparedStatement insertStatement = connection.prepareStatement(insert);){
                this.writeHistoryEntries(insertStatement, playerId, history);
                insertStatement.executeBatch();
            }
            connection.commit();
        }
        catch (SQLException ex) {
            this.logger.log(Level.SEVERE, String.format("Failed to save history for player %s to MySQL.", playerId), ex);
        }
        finally {
            this.historyWriteLock.unlock();
        }
    }

    @Override
    public void close() {
    }

    private Connection getConnection() throws SQLException {
        return DriverManager.getConnection(this.jdbcUrl, this.mysql.username(), this.mysql.password());
    }

    private UUID parseUuid(String raw) {
        if (raw == null || raw.isEmpty()) {
            return null;
        }
        try {
            return UUID.fromString(raw);
        }
        catch (IllegalArgumentException ex) {
            this.logger.log(Level.WARNING, "Invalid UUID encountered in MySQL storage: {0}", raw);
            return null;
        }
    }

    private AuctionTransactionType parseType(String typeName) {
        if (typeName == null || typeName.isEmpty()) {
            return null;
        }
        try {
            return AuctionTransactionType.valueOf(typeName.trim().toUpperCase());
        }
        catch (IllegalArgumentException ex) {
            this.logger.log(Level.WARNING, "Invalid transaction type encountered in MySQL storage: {0}", typeName);
            return null;
        }
    }

    private void writeHistoryEntries(PreparedStatement statement, UUID playerId, Deque<AuctionTransactionHistoryEntry> history) throws SQLException {
        if (history == null || history.isEmpty()) {
            return;
        }
        int index = 0;
        for (AuctionTransactionHistoryEntry entry : history) {
            statement.setString(1, playerId.toString());
            statement.setInt(2, index++);
            statement.setString(3, entry.type().name());
            statement.setLong(4, entry.timestamp());
            statement.setDouble(5, entry.price());
            if (entry.counterpartId() != null) {
                statement.setString(6, entry.counterpartId().toString());
            } else {
                statement.setNull(6, 12);
            }
            if (entry.counterpartName() != null && !entry.counterpartName().isEmpty()) {
                statement.setString(7, entry.counterpartName());
            } else {
                statement.setNull(7, 12);
            }
            statement.setString(8, ItemStackSerialization.serialize(entry.item(), this.logger));
            statement.addBatch();
        }
    }

    private String sanitize(String input) {
        if (input == null || input.isBlank()) {
            return "ezauction_table";
        }
        String sanitized = input.replaceAll("[^a-zA-Z0-9_]+", "_");
        if (sanitized.isBlank()) {
            sanitized = "ezauction_table";
        }
        if (sanitized.length() > 64) {
            sanitized = sanitized.substring(0, 64);
        }
        return sanitized;
    }
}

