/*
 * Decompiled with CFR 0.152.
 */
package com.skyblockexp.ezauction.hologram;

import com.skyblockexp.ezauction.AuctionManager;
import com.skyblockexp.ezauction.compat.HologramDisplay;
import com.skyblockexp.ezauction.compat.HologramPlatform;
import com.skyblockexp.ezauction.config.AuctionHologramConfiguration;
import com.skyblockexp.ezauction.hologram.AuctionHologramType;
import com.skyblockexp.ezauction.transaction.AuctionTransactionService;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.logging.Level;
import net.kyori.adventure.text.Component;
import org.bukkit.Location;
import org.bukkit.World;
import org.bukkit.entity.Entity;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitTask;

public final class AuctionHologramManager {
    private final JavaPlugin plugin;
    private final AuctionManager auctionManager;
    private final AuctionTransactionService transactionService;
    private final AuctionHologramConfiguration configuration;
    private final HologramPlatform hologramPlatform;
    private final Map<UUID, HologramEntry> trackedDisplays = new HashMap<UUID, HologramEntry>();
    private String markerKey;
    private BukkitTask updateTask;

    public AuctionHologramManager(JavaPlugin plugin, AuctionManager auctionManager, AuctionTransactionService transactionService, AuctionHologramConfiguration configuration, HologramPlatform hologramPlatform) {
        this.plugin = Objects.requireNonNull(plugin, "plugin");
        this.auctionManager = Objects.requireNonNull(auctionManager, "auctionManager");
        this.transactionService = Objects.requireNonNull(transactionService, "transactionService");
        this.configuration = configuration != null ? configuration : AuctionHologramConfiguration.defaults();
        this.hologramPlatform = Objects.requireNonNull(hologramPlatform, "hologramPlatform");
    }

    public void enable() {
        if (!this.hologramPlatform.isSupported()) {
            return;
        }
        this.markerKey = "auction_hologram_type";
        this.attachExistingDisplays();
        long interval = this.configuration.updateIntervalTicks();
        this.updateTask = this.plugin.getServer().getScheduler().runTaskTimer((Plugin)this.plugin, this::updateTrackedDisplays, 40L, Math.max(20L, interval));
    }

    public void disable() {
        if (this.updateTask != null) {
            this.updateTask.cancel();
            this.updateTask = null;
        }
        this.trackedDisplays.clear();
        this.markerKey = null;
    }

    public boolean ensureHologram(Location location, AuctionHologramType type) {
        int chunkZ;
        if (location == null || type == null) {
            return false;
        }
        World world = location.getWorld();
        if (world == null) {
            return false;
        }
        Location spawnLocation = this.normalizeLocation(location.clone());
        spawnLocation.add(0.0, this.configuration.heightOffset(), 0.0);
        HologramDisplay existing = this.findExistingDisplay(spawnLocation, type);
        if (existing != null) {
            this.registerDisplay(existing, type);
            this.updateDisplay(existing, type);
            return true;
        }
        int chunkX = spawnLocation.getBlockX() >> 4;
        if (!world.isChunkLoaded(chunkX, chunkZ = spawnLocation.getBlockZ() >> 4)) {
            world.loadChunk(chunkX, chunkZ);
            if (!world.isChunkLoaded(chunkX, chunkZ)) {
                return false;
            }
        }
        if (this.trackedDisplays.size() >= this.configuration.maxHolograms()) {
            this.plugin.getLogger().warning("Max auction hologram count reached (" + this.configuration.maxHolograms() + "). Placement denied.");
            return false;
        }
        HologramDisplay created = this.hologramPlatform.spawn(spawnLocation);
        if (created == null) {
            return false;
        }
        this.hologramPlatform.configureBase(created);
        this.markDisplay(created, type);
        this.registerDisplay(created, type);
        this.updateDisplay(created, type);
        return true;
    }

    public boolean removeNearest(Location location) {
        if (location == null) {
            return false;
        }
        World world = location.getWorld();
        if (world == null) {
            return false;
        }
        Location searchLocation = this.normalizeLocation(location.clone());
        double radius = this.configuration.searchRadius();
        Collection<HologramDisplay> nearby = this.hologramPlatform.findNearby(searchLocation, radius);
        HologramDisplay closest = null;
        double closestDistance = Double.MAX_VALUE;
        for (HologramDisplay display : nearby) {
            double distance;
            AuctionHologramType type = this.readType(display);
            if (type == null || !((distance = display.location().distanceSquared(searchLocation)) < closestDistance)) continue;
            closestDistance = distance;
            closest = display;
        }
        if (closest == null) {
            return false;
        }
        this.trackedDisplays.remove(closest.uniqueId());
        closest.remove();
        return true;
    }

    public Map<AuctionHologramType, Integer> trackedCounts() {
        EnumMap<AuctionHologramType, Integer> counts = new EnumMap<AuctionHologramType, Integer>(AuctionHologramType.class);
        for (HologramEntry entry : this.trackedDisplays.values()) {
            counts.merge(entry.type(), 1, Integer::sum);
        }
        return counts;
    }

    private void attachExistingDisplays() {
        for (World world : this.plugin.getServer().getWorlds()) {
            for (HologramDisplay display : this.hologramPlatform.findDisplays(world)) {
                AuctionHologramType type = this.readType(display);
                if (type == null) continue;
                this.hologramPlatform.configureBase(display);
                this.registerDisplay(display, type);
            }
        }
    }

    private void updateTrackedDisplays() {
        if (this.trackedDisplays.isEmpty()) {
            return;
        }
        if (this.configuration.batchUpdate()) {
            this.trackedDisplays.entrySet().removeIf(entry -> {
                HologramDisplay display = ((HologramEntry)entry.getValue()).display();
                if (!this.isDisplayValid(display)) {
                    return true;
                }
                try {
                    this.updateDisplay(display, ((HologramEntry)entry.getValue()).type());
                }
                catch (Exception ex) {
                    this.plugin.getLogger().log(Level.WARNING, "Failed to update auction hologram display.", ex);
                }
                return false;
            });
        } else {
            int batchSize = Math.max(1, this.trackedDisplays.size() / 10);
            int count = 0;
            for (Map.Entry<UUID, HologramEntry> entry2 : this.trackedDisplays.entrySet()) {
                HologramDisplay display = entry2.getValue().display();
                if (!this.isDisplayValid(display)) {
                    this.trackedDisplays.remove(entry2.getKey());
                    continue;
                }
                try {
                    this.updateDisplay(display, entry2.getValue().type());
                }
                catch (Exception ex) {
                    this.plugin.getLogger().log(Level.WARNING, "Failed to update auction hologram display.", ex);
                }
                if (++count < batchSize) continue;
                break;
            }
        }
    }

    private void updateDisplay(HologramDisplay display, AuctionHologramType type) {
        if (display == null || type == null) {
            return;
        }
        Component text = type.render(this.auctionManager, this.transactionService);
        if (text != null) {
            display.setText(text);
        }
    }

    private void registerDisplay(HologramDisplay display, AuctionHologramType type) {
        if (display == null || type == null) {
            return;
        }
        this.trackedDisplays.put(display.uniqueId(), new HologramEntry(display, type));
        this.markDisplay(display, type);
        this.hologramPlatform.configureBase(display);
    }

    private void markDisplay(HologramDisplay display, AuctionHologramType type) {
        if (display == null || type == null || this.markerKey == null) {
            return;
        }
        this.hologramPlatform.setMarker(display, this.markerKey, type.name());
    }

    public AuctionHologramConfiguration getConfiguration() {
        return this.configuration;
    }

    public boolean isHologramEntity(Entity entity) {
        return this.hologramPlatform.isDisplayEntity(entity);
    }

    public AuctionHologramType readType(Entity entity) {
        HologramDisplay display = this.hologramPlatform.wrap(entity);
        return this.readType(display);
    }

    public AuctionHologramType readType(HologramDisplay display) {
        if (display == null || this.markerKey == null) {
            return null;
        }
        String raw = this.hologramPlatform.getMarker(display, this.markerKey);
        if (raw == null || raw.isBlank()) {
            return null;
        }
        try {
            return AuctionHologramType.valueOf(raw.trim().toUpperCase(Locale.ENGLISH));
        }
        catch (IllegalArgumentException ex) {
            return AuctionHologramType.fromName(raw);
        }
    }

    private HologramDisplay findExistingDisplay(Location location, AuctionHologramType type) {
        if (location == null || type == null) {
            return null;
        }
        World world = location.getWorld();
        if (world == null) {
            return null;
        }
        double radius = this.configuration.searchRadius();
        Collection<HologramDisplay> nearby = this.hologramPlatform.findNearby(location, radius);
        for (HologramDisplay display : nearby) {
            AuctionHologramType existingType = this.readType(display);
            if (existingType != type) continue;
            return display;
        }
        return null;
    }

    private boolean isDisplayValid(HologramDisplay display) {
        return display != null && display.isValid();
    }

    private Location normalizeLocation(Location location) {
        Location clone = location.clone();
        clone.setX((double)clone.getBlockX() + 0.5);
        clone.setZ((double)clone.getBlockZ() + 0.5);
        clone.setPitch(0.0f);
        clone.setYaw(0.0f);
        return clone;
    }

    private record HologramEntry(HologramDisplay display, AuctionHologramType type) {
    }
}

