/*
 * Decompiled with CFR 0.152.
 */
package com.ryderbelserion.fusion.core.files;

import ch.jalu.configme.SettingsManagerBuilder;
import ch.jalu.configme.resource.YamlFileResourceOptions;
import com.ryderbelserion.fusion.core.FusionCore;
import com.ryderbelserion.fusion.core.api.exceptions.FusionException;
import com.ryderbelserion.fusion.core.files.enums.FileAction;
import com.ryderbelserion.fusion.core.files.enums.FileType;
import com.ryderbelserion.fusion.core.files.interfaces.ICustomFile;
import com.ryderbelserion.fusion.core.files.interfaces.IFileManager;
import com.ryderbelserion.fusion.core.files.types.JaluCustomFile;
import com.ryderbelserion.fusion.core.files.types.JsonCustomFile;
import com.ryderbelserion.fusion.core.files.types.LogCustomFile;
import com.ryderbelserion.fusion.core.files.types.YamlCustomFile;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class FileManager
extends IFileManager<FileManager> {
    protected final FusionCore fusion;
    protected final Path dataPath;
    protected final Map<Path, ICustomFile<?, ?, ?, ?>> files = new HashMap();

    public FileManager(@NotNull FusionCore fusion) {
        this.fusion = fusion;
        this.dataPath = this.fusion.getDataPath();
    }

    @Override
    @NotNull
    public FileManager addFolder(@NotNull Path folder, @NotNull FileType fileType, @NotNull Consumer<ICustomFile<?, ?, ?, ?>> consumer) {
        this.extractFolder(folder.getFileName().toString(), folder.getParent());
        for (Path path : this.fusion.getFiles(folder, ".yml", this.fusion.getDepth())) {
            this.addFile(path, fileType, (Consumer)consumer);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager addFolder(@NotNull Path folder, @NotNull Consumer<YamlFileResourceOptions.Builder> options, @NotNull Consumer<SettingsManagerBuilder> builder) {
        for (Path path : this.fusion.getFiles(folder, ".yml", this.fusion.getDepth())) {
            this.addFile(path, (Consumer)options, (Consumer)builder);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager addFile(@NotNull Path path, @NotNull Consumer<YamlFileResourceOptions.Builder> options, @NotNull Consumer<SettingsManagerBuilder> builder) {
        if (this.files.containsKey(path)) {
            this.files.get(path).load();
            return this;
        }
        JaluCustomFile customFile = this.buildJaluFile(path, options, builder);
        this.files.putIfAbsent(path, customFile);
        return this;
    }

    @Override
    @NotNull
    public FileManager addFile(@NotNull Path path, @NotNull FileType fileType, @NotNull Consumer<ICustomFile<?, ?, ?, ?>> consumer) {
        if (this.files.containsKey(path)) {
            ICustomFile<?, ?, ?, ?> customFile = this.files.get(path);
            consumer.accept(customFile);
            if (!customFile.hasAction(FileAction.FILE_ALREADY_RELOADED)) {
                customFile.load();
                return this;
            }
            this.fusion.log("info", "Path {} already exists in the cache, so we don't need to rebuild it", path);
            return this;
        }
        ICustomFile customFile = null;
        switch (fileType) {
            case YAML: 
            case FUSION_YAML: {
                customFile = this.buildYamlFile(path, consumer::accept);
                break;
            }
            case JSON: 
            case FUSION_JSON: {
                customFile = this.buildJsonFile(path, consumer::accept);
                break;
            }
            case LOG: {
                customFile = this.buildLogFile(path, consumer::accept);
            }
        }
        if (customFile == null) {
            return this;
        }
        this.files.putIfAbsent(path, customFile);
        return this;
    }

    @Override
    @NotNull
    public FileManager removeFile(@NotNull Path path) {
        Optional<ICustomFile<?, ?, ?, ?>> variable = this.getFile(path);
        if (variable.isEmpty()) {
            return this;
        }
        ICustomFile<?, ?, ?, ?> customFile = variable.get();
        FileType fileType = customFile.getFileType();
        if (fileType == FileType.FUSION_YAML || fileType == FileType.FUSION_JSON) {
            return this;
        }
        this.files.remove(path);
        return this;
    }

    @Override
    @NotNull
    public FileManager purge() {
        HashMap files = new HashMap(this.files);
        for (Map.Entry entry : files.entrySet()) {
            this.removeFile((Path)entry.getKey());
        }
        return this;
    }

    @NotNull
    public FileManager addFile(@NotNull Path path, @NotNull ICustomFile<?, ?, ?, ?> customFile) {
        this.files.putIfAbsent(path, customFile);
        return this;
    }

    @Override
    @NotNull
    public YamlCustomFile buildYamlFile(@NotNull Path path, @NotNull Consumer<YamlCustomFile> consumer) {
        return (YamlCustomFile)new YamlCustomFile(this, path, consumer).load();
    }

    @Override
    @NotNull
    public JsonCustomFile buildJsonFile(@NotNull Path path, @NotNull Consumer<JsonCustomFile> consumer) {
        return (JsonCustomFile)new JsonCustomFile(this, path, consumer).load();
    }

    @Override
    @NotNull
    public JaluCustomFile buildJaluFile(@NotNull Path path, @NotNull Consumer<YamlFileResourceOptions.Builder> options, @NotNull Consumer<SettingsManagerBuilder> builder) {
        return (JaluCustomFile)new JaluCustomFile(this, path, options, builder).load();
    }

    @Override
    @NotNull
    public LogCustomFile buildLogFile(@NotNull Path path, @NotNull Consumer<LogCustomFile> consumer) {
        return (LogCustomFile)new LogCustomFile(this, path, consumer).load();
    }

    @Override
    @NotNull
    public FileManager reloadFile(@NotNull Path path) {
        Optional<ICustomFile<?, ?, ?, ?>> customFile = this.getFile(path);
        if (customFile.isEmpty()) {
            return this;
        }
        customFile.get().load();
        return this;
    }

    @Override
    @NotNull
    public FileManager saveFile(@NotNull Path path) {
        Optional<ICustomFile<?, ?, ?, ?>> customFile = this.getFile(path);
        if (customFile.isEmpty()) {
            return this;
        }
        customFile.get().save();
        return this;
    }

    @Override
    @NotNull
    public Optional<ICustomFile<?, ?, ?, ?>> getFile(@NotNull Path path) {
        return Optional.ofNullable(this.files.get(path));
    }

    @Override
    @NotNull
    public FileManager extractFolder(@NotNull String folder, @NotNull Path output) {
        Path path = output.resolve(folder);
        if (Files.exists(path, new LinkOption[0])) {
            this.fusion.log("info", "Cannot extract folder {} to {}, because it already exists @ {}.", folder, output, path);
            return this;
        }
        if (Files.notExists(path, new LinkOption[0])) {
            try {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            catch (Exception exception) {
                throw new FusionException("Failed to create %s".formatted(path), exception);
            }
        }
        try (JarFile jarFile = new JarFile(Path.of(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).toFile());){
            Set<JarEntry> entries = jarFile.stream().filter(entry -> !entry.getName().endsWith(".class")).filter(entry -> !entry.getName().startsWith("META-INF")).filter(entry -> !entry.isDirectory()).filter(entry -> entry.getName().startsWith(folder)).collect(Collectors.toSet());
            entries.forEach(entry -> {
                Path target = output.resolve(entry.getName());
                Path parent = target.getParent();
                if (!Files.exists(parent, new LinkOption[0])) {
                    try {
                        Files.createDirectories(parent, new FileAttribute[0]);
                    }
                    catch (IOException exception) {
                        throw new FusionException("Failed to create %s".formatted(parent), exception);
                    }
                }
                if (Files.notExists(target, new LinkOption[0])) {
                    try (InputStream stream = jarFile.getInputStream((ZipEntry)entry);){
                        Files.copy(stream, target, new CopyOption[0]);
                    }
                    catch (IOException exception) {
                        throw new FusionException("Failed to copy %s to %s".formatted(target, parent), exception);
                    }
                }
            });
        }
        catch (IOException | URISyntaxException exception) {
            throw new FusionException("Failed to extract folder %s".formatted(path), exception);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager extractFile(@NotNull Path path) {
        if (Files.exists(path, new LinkOption[0])) {
            this.fusion.log("info", "Cannot extract {} to {}, because it already exists.", path.getFileName().toString(), path);
            return this;
        }
        String fileName = path.getFileName().toString();
        try (JarFile jarFile = new JarFile(Path.of(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).toFile());){
            Set<JarEntry> entries = jarFile.stream().filter(entry -> !entry.getName().endsWith(".class")).filter(entry -> !entry.getName().startsWith("META-INF")).filter(entry -> !entry.isDirectory()).filter(entry -> entry.getName().equalsIgnoreCase(fileName)).collect(Collectors.toSet());
            entries.forEach(entry -> {
                if (Files.notExists(path, new LinkOption[0])) {
                    try (InputStream stream = jarFile.getInputStream((ZipEntry)entry);){
                        Files.copy(stream, path, new CopyOption[0]);
                    }
                    catch (IOException exception) {
                        throw new FusionException("Failed to copy %s to %s".formatted(entry.getName(), path), exception);
                    }
                }
            });
        }
        catch (IOException | URISyntaxException exception) {
            throw new FusionException("Failed to extract file %s".formatted(path), exception);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager compressFolder(@NotNull Path path, @NotNull String content) {
        if (!Files.exists(path, new LinkOption[0])) {
            return this;
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            return this;
        }
        Path target = this.dataPath.resolve(this.asString(path, content));
        try (ZipOutputStream output = new ZipOutputStream(Files.newOutputStream(target, new OpenOption[0]));
             Stream<Path> values = Files.walk(path, new FileVisitOption[0]);){
            List<Path> entries = values.filter(key -> !Files.isDirectory(key, new LinkOption[0])).toList();
            for (Path entry : entries) {
                if (Files.size(entry) <= 0L) continue;
                ZipEntry zipEntry = new ZipEntry(entry.toString());
                output.putNextEntry(zipEntry);
                Files.copy(entry, output);
                output.closeEntry();
            }
        }
        catch (IOException exception) {
            throw new FusionException("Failed to compress folder %s".formatted(path), exception);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager compressFile(@NotNull Path path, @Nullable Path folder, @NotNull String content) {
        long size;
        if (!Files.exists(path, new LinkOption[0])) {
            return this;
        }
        try {
            size = Files.size(path);
        }
        catch (Exception exception) {
            throw new FusionException("Failed to calculate file size for %s".formatted(path), exception);
        }
        if (size <= 0L) {
            return this;
        }
        String builder = this.asString(path, content);
        Path target = folder == null ? this.dataPath : folder.resolve(builder);
        try (ZipOutputStream output = new ZipOutputStream(Files.newOutputStream(target, new OpenOption[0]));){
            ZipEntry entry = new ZipEntry(path.getFileName().toString());
            output.putNextEntry(entry);
            Files.copy(path, output);
            output.closeEntry();
        }
        catch (Exception exception) {
            throw new FusionException("Failed to compress %s".formatted(path), exception);
        }
        return this;
    }

    @Override
    @NotNull
    public FileManager writeFile(@NotNull Path path, @NotNull String content) {
        try {
            Files.writeString(path, (CharSequence)content, StandardOpenOption.APPEND);
        }
        catch (IOException exception) {
            throw new FusionException("Failed to write %s to %s".formatted(content, path), exception);
        }
        return this;
    }

    @Override
    public final int getFileCount(@NotNull Path path, @NotNull String extension) {
        return this.fusion.getFiles(path, extension, 1).size();
    }

    @Override
    public final boolean hasFile(@NotNull Path path) {
        return this.files.containsKey(path);
    }

    @NotNull
    public final Map<Path, ICustomFile<?, ?, ?, ?>> getFiles() {
        return Collections.unmodifiableMap(this.files);
    }

    @Override
    @NotNull
    public final FileManager refresh(boolean save) {
        if (this.files.isEmpty()) {
            return this;
        }
        ArrayList<Path> keys = new ArrayList<Path>();
        for (Map.Entry<Path, ICustomFile<?, ?, ?, ?>> file : this.files.entrySet()) {
            ICustomFile<?, ?, ?, ?> value = file.getValue();
            if (value == null) continue;
            Path path = value.getPath();
            if (!Files.exists(path, new LinkOption[0])) {
                keys.add(file.getKey());
                continue;
            }
            if (save) {
                value.save();
                continue;
            }
            value.load();
        }
        if (!keys.isEmpty()) {
            keys.forEach(this.files::remove);
        }
        return this;
    }

    private String asString(@NotNull Path path, @NotNull String content) {
        StringBuilder builder = new StringBuilder();
        builder.append(LocalDate.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        if (!content.isEmpty()) {
            builder.append(content);
        }
        int fileCount = this.getFileCount(path, ".gz");
        builder.append("-").append(fileCount).append(".gz");
        return builder.toString();
    }
}

