/*
 * Decompiled with CFR 0.152.
 */
package de.bluecolored.bluenbt;

import de.bluecolored.bluenbt.DataLogInputStream;
import de.bluecolored.bluenbt.TagType;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Objects;
import java.util.function.IntFunction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class NBTReader
implements Closeable {
    private static final String UNKNOWN_NAME = "<unknown>";
    private final DataLogInputStream log;
    private final DataInputStream in;
    private int stackPosition = 0;
    private TagType[] stack = new TagType[32];
    private String[] nameStack = new String[32];
    private int[] listStack = new int[32];

    public NBTReader(byte @NotNull [] data) {
        this(new ByteArrayInputStream(data));
    }

    public NBTReader(@NotNull InputStream in) {
        Objects.requireNonNull(in);
        this.log = new DataLogInputStream(in);
        this.in = new DataInputStream(this.log);
    }

    public TagType peek() throws IOException {
        TagType peek = this.stack[this.stackPosition];
        if (peek == null) {
            this.stack[this.stackPosition] = peek = this.readTag();
        }
        return peek;
    }

    public String name() throws IOException {
        String name = this.nameStack[this.stackPosition];
        if (name == null) {
            name = this.peek() != TagType.END ? this.in.readUTF() : UNKNOWN_NAME;
            this.nameStack[this.stackPosition] = name;
        }
        return name;
    }

    public void beginCompound() throws IOException {
        this.checkState(TagType.COMPOUND);
        this.advanceStack();
    }

    public void endCompound() throws IOException {
        this.checkState(TagType.END);
        if (!this.inCompound()) {
            throw new IllegalStateException("Can not end compound. Current element is not in a compound! At: " + this.path());
        }
        this.reduceStack();
        this.next();
    }

    public int beginList() throws IOException {
        this.checkState(TagType.LIST);
        this.advanceStack();
        TagType listType = this.readTag();
        int listLength = this.in.readInt();
        this.stack[this.stackPosition] = listLength == 0 ? TagType.END : listType;
        this.listStack[this.stackPosition] = listLength;
        this.nameStack[this.stackPosition] = UNKNOWN_NAME;
        return listLength;
    }

    public void endList() throws IOException {
        this.checkState(TagType.END);
        if (!this.inList()) {
            throw new IllegalStateException("Can not end list. Current element is not in a list! At: " + this.path());
        }
        this.reduceStack();
        this.next();
    }

    public boolean hasNext() throws IOException {
        return this.peek() != TagType.END;
    }

    public byte nextByte() throws IOException {
        this.checkState(TagType.BYTE);
        this.next();
        return this.in.readByte();
    }

    public short nextShort() throws IOException {
        this.checkState(TagType.SHORT);
        this.next();
        return this.in.readShort();
    }

    public int nextInt() throws IOException {
        this.checkState(TagType.INT);
        this.next();
        return this.in.readInt();
    }

    public long nextLong() throws IOException {
        this.checkState(TagType.LONG);
        this.next();
        return this.in.readLong();
    }

    public float nextFloat() throws IOException {
        this.checkState(TagType.FLOAT);
        this.next();
        return this.in.readFloat();
    }

    public double nextDouble() throws IOException {
        this.checkState(TagType.DOUBLE);
        this.next();
        return this.in.readDouble();
    }

    public String nextString() throws IOException {
        this.checkState(TagType.STRING);
        this.next();
        return this.in.readUTF();
    }

    public byte[] nextByteArray() throws IOException {
        this.checkState(TagType.BYTE_ARRAY);
        this.next();
        byte[] data = new byte[this.in.readInt()];
        this.in.readFully(data);
        return data;
    }

    public int nextByteArray(byte[] buffer) throws IOException {
        this.checkState(TagType.BYTE_ARRAY);
        this.next();
        int length = this.in.readInt();
        int readLength = Math.min(length, buffer.length);
        this.in.readFully(buffer, 0, readLength);
        this.skipNBytes(length - readLength);
        return length;
    }

    public int[] nextIntArray() throws IOException {
        this.checkState(TagType.INT_ARRAY);
        this.next();
        int[] data = new int[this.in.readInt()];
        for (int i = 0; i < data.length; ++i) {
            data[i] = this.in.readInt();
        }
        return data;
    }

    public int nextIntArray(int[] buffer) throws IOException {
        this.checkState(TagType.INT_ARRAY);
        this.next();
        int length = this.in.readInt();
        int readLength = Math.min(length, buffer.length);
        for (int i = 0; i < readLength; ++i) {
            buffer[i] = this.in.readInt();
        }
        this.skipNBytes((long)(length - readLength) * (long)TagType.INT.getSize());
        return length;
    }

    public long[] nextLongArray() throws IOException {
        this.checkState(TagType.LONG_ARRAY);
        this.next();
        long[] data = new long[this.in.readInt()];
        for (int i = 0; i < data.length; ++i) {
            data[i] = this.in.readLong();
        }
        return data;
    }

    public int nextLongArray(long[] buffer) throws IOException {
        this.checkState(TagType.LONG_ARRAY);
        this.next();
        int length = this.in.readInt();
        int readLength = Math.min(length, buffer.length);
        for (int i = 0; i < readLength; ++i) {
            buffer[i] = this.in.readLong();
        }
        this.skipNBytes((long)(length - readLength) * (long)TagType.LONG.getSize());
        return length;
    }

    public byte[] nextArrayAsByteArray() throws IOException {
        if (this.peek() == TagType.BYTE_ARRAY) {
            return this.nextByteArray();
        }
        return this.nextArray(byte[]::new);
    }

    public int[] nextArrayAsIntArray() throws IOException {
        if (this.peek() == TagType.INT_ARRAY) {
            return this.nextIntArray();
        }
        return this.nextArray(int[]::new);
    }

    public long[] nextArrayAsLongArray() throws IOException {
        if (this.peek() == TagType.LONG_ARRAY) {
            return this.nextLongArray();
        }
        return this.nextArray(long[]::new);
    }

    public int nextArray(Object bufferArray) throws IOException {
        this.checkState();
        TagType type = this.peek();
        int length = this.in.readInt();
        switch (type) {
            case BYTE_ARRAY: {
                this.readByteArray(length, bufferArray);
                break;
            }
            case INT_ARRAY: {
                this.readIntArray(length, bufferArray);
                break;
            }
            case LONG_ARRAY: {
                this.readLongArray(length, bufferArray);
                break;
            }
            default: {
                throw new IllegalStateException("Expected any array-type but got " + String.valueOf((Object)this.peek()) + ". At: " + this.path());
            }
        }
        this.next();
        return length;
    }

    public <A> A nextArray(IntFunction<A> generator) throws IOException {
        this.checkState();
        TagType type = this.peek();
        int length = this.in.readInt();
        A bufferArray = generator.apply(length);
        switch (type) {
            case BYTE_ARRAY: {
                this.readByteArray(length, bufferArray);
                break;
            }
            case INT_ARRAY: {
                this.readIntArray(length, bufferArray);
                break;
            }
            case LONG_ARRAY: {
                this.readLongArray(length, bufferArray);
                break;
            }
            default: {
                throw new IllegalStateException("Expected any array-type but got " + String.valueOf((Object)this.peek()) + ". At: " + this.path());
            }
        }
        this.next();
        return bufferArray;
    }

    private void readByteArray(int length, Object bufferArray) throws IOException {
        this.checkState(TagType.BYTE_ARRAY);
        int readLength = Math.min(length, Array.getLength(bufferArray));
        for (int i = 0; i < readLength; ++i) {
            Array.setByte(bufferArray, i, this.in.readByte());
        }
        this.skipNBytes(length - readLength);
    }

    private void readIntArray(int length, Object bufferArray) throws IOException {
        this.checkState(TagType.INT_ARRAY);
        int readLength = Math.min(length, Array.getLength(bufferArray));
        for (int i = 0; i < readLength; ++i) {
            Array.setInt(bufferArray, i, this.in.readInt());
        }
        this.skipNBytes((long)(length - readLength) * (long)TagType.INT.getSize());
    }

    private void readLongArray(int length, Object bufferArray) throws IOException {
        this.checkState(TagType.LONG_ARRAY);
        int readLength = Math.min(length, Array.getLength(bufferArray));
        for (int i = 0; i < readLength; ++i) {
            Array.setLong(bufferArray, i, this.in.readLong());
        }
        this.skipNBytes((long)(length - readLength) * (long)TagType.LONG.getSize());
    }

    public byte[] raw() throws IOException {
        this.checkState();
        this.log.startLog();
        DataOutputStream dOut = new DataOutputStream(this.log.log);
        dOut.write(this.peek().getId());
        dOut.writeUTF(this.name());
        dOut.flush();
        this.skip();
        return this.log.stopLog();
    }

    public void skip() throws IOException {
        this.skip(0);
    }

    public void skip(int out) throws IOException {
        if (out < 0) {
            throw new IllegalArgumentException("'out' can not be negative!");
        }
        if (out == 0 && this.peek() == TagType.END) {
            throw new IllegalStateException("Can not skip END tag!");
        }
        do {
            TagType type = this.peek();
            switch (type) {
                case END: {
                    if (this.inList()) {
                        this.endList();
                    } else {
                        this.endCompound();
                    }
                    --out;
                    break;
                }
                case BYTE: 
                case SHORT: 
                case INT: 
                case LONG: 
                case FLOAT: 
                case DOUBLE: {
                    this.checkState();
                    this.skipNBytes(type.getSize());
                    this.next();
                    break;
                }
                case STRING: {
                    this.checkState();
                    this.skipUTF();
                    this.next();
                    break;
                }
                case BYTE_ARRAY: {
                    this.checkState();
                    long length = this.in.readInt();
                    this.skipNBytes((long)TagType.BYTE.getSize() * length);
                    this.next();
                    break;
                }
                case INT_ARRAY: {
                    this.checkState();
                    long length = this.in.readInt();
                    this.skipNBytes((long)TagType.INT.getSize() * length);
                    this.next();
                    break;
                }
                case LONG_ARRAY: {
                    this.checkState();
                    long length = this.in.readInt();
                    this.skipNBytes((long)TagType.LONG.getSize() * length);
                    this.next();
                    break;
                }
                case COMPOUND: {
                    this.beginCompound();
                    ++out;
                    break;
                }
                case LIST: {
                    int length = this.beginList();
                    TagType listType = this.peek();
                    ++out;
                    if (listType.getSize() == -1) break;
                    this.in.skipBytes(listType.getSize() * length);
                    this.listStack[this.stackPosition] = 0;
                    this.stack[this.stackPosition] = TagType.END;
                    break;
                }
            }
        } while (out > 0);
    }

    public int remainingListItems() {
        return this.listStack[this.stackPosition];
    }

    public boolean inCompound() {
        return this.stackPosition > 0 && this.stack[this.stackPosition - 1] == TagType.COMPOUND;
    }

    public boolean inList() {
        return this.stackPosition > 0 && this.stack[this.stackPosition - 1] == TagType.LIST;
    }

    public String path() throws IOException {
        this.checkState();
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i <= this.stackPosition; ++i) {
            if (i > 1) {
                if (this.stack[i - 1] == TagType.LIST) {
                    sb.append('[').append(this.listStack[i]).append(']');
                    continue;
                }
                sb.append('.').append(this.nameStack[i]);
                continue;
            }
            sb.append(this.nameStack[i]);
        }
        return sb.toString();
    }

    private void next() {
        if (this.inList()) {
            int n = this.stackPosition;
            this.listStack[n] = this.listStack[n] - 1;
            if (this.listStack[this.stackPosition] == 0) {
                this.stack[this.stackPosition] = TagType.END;
            }
        } else {
            this.stack[this.stackPosition] = null;
            this.nameStack[this.stackPosition] = null;
        }
    }

    private void advanceStack() {
        ++this.stackPosition;
        if (this.stackPosition == this.stack.length) {
            int newLength = this.stack.length * 2;
            this.stack = Arrays.copyOf(this.stack, newLength);
            this.nameStack = Arrays.copyOf(this.nameStack, newLength);
            this.listStack = Arrays.copyOf(this.listStack, newLength);
        }
        this.stack[this.stackPosition] = null;
        this.nameStack[this.stackPosition] = null;
        this.listStack[this.stackPosition] = 0;
    }

    private void reduceStack() {
        if (this.stackPosition == 0) {
            throw new IllegalStateException("Can not reduce empty stack!");
        }
        --this.stackPosition;
    }

    private TagType readTag() throws IOException {
        int tagId = this.in.read();
        if (tagId == -1) {
            throw new EOFException();
        }
        return TagType.forId(tagId);
    }

    private void skipUTF() throws IOException {
        int length = this.in.readUnsignedShort();
        this.skipNBytes(length);
    }

    private void skipNBytes(long n) throws IOException {
        while (n > 0L) {
            long ns = this.in.skip(n);
            if (ns > 0L && ns <= n) {
                n -= ns;
                continue;
            }
            if (ns == 0L) {
                if (this.in.read() == -1) {
                    throw new EOFException();
                }
                --n;
                continue;
            }
            throw new IOException("Unable to skip exactly");
        }
    }

    private void checkState() throws IOException {
        this.checkState(null);
    }

    private void checkState(@Nullable TagType expected) throws IOException {
        TagType type = this.peek();
        if (expected != null && type != expected) {
            throw new IllegalStateException("Expected type " + String.valueOf((Object)expected) + " but got " + String.valueOf((Object)this.peek()) + ". At: " + this.path());
        }
        if (this.nameStack[this.stackPosition] == null) {
            this.nameStack[this.stackPosition] = UNKNOWN_NAME;
            if (type != TagType.END) {
                this.skipUTF();
            }
        }
    }

    @Override
    public void close() throws IOException {
        this.in.close();
    }
}

